;=============================================================================
; (C) 1999 by Helge Wagner
;
; FILE: LANC_IF.ASM
;
; DESC: Sony LANC Protocol low level interface
;
; BY:   H.Wagner, mailto:<H.Wagner@sbs-or.de>
; DATE: 04.03.1999
;
; NOTE: Tab width=4
;=============================================================================
; Rev  Date       Author   Description
;-----------------------------------------------------------------------------
; 1.01 08.12.1999 H.Wagner -Changed to easy adapt to other interfaces.
;=============================================================================

.MODEL  SMALL,SYSCALL
 PAGE   60,132
 TITLE  LANC_IF.ASM
 OPTION PROC:private

.nolist
  include lanc_if.inc
  include lanc_if.inp
.list

;IRDEO_LANC_INTERFACE        EQU
PARPORT_LANC_INTERFACE     EQU

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%
;%                MAIN MODULE
;%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

; this is for IRdeo

IFDEF IRDEO_LANC_INTERFACE
SEND_PORT       TEXTEQU <ser_port_lcr>
RECEIVE_PORT    TEXTEQU <ser_port_msr>
SEND_HIGH       EQU 01000011b
SEND_LOW        EQU 00000011b
RECV_MASK       EQU 10h
RECV_HIGH       EQU 10h
RECV_LOW        EQU 00h
ENDIF ;IFDEF IRDEO_LANC_INTERFACE

; this is for the parallel port LANC (Pins 15,17,18)
IFDEF PARPORT_LANC_INTERFACE
SEND_PORT       TEXTEQU <par_port_controlreg>
RECEIVE_PORT    TEXTEQU <par_port_statusreg>
SEND_HIGH       EQU 00h
SEND_LOW        EQU 08h
RECV_MASK       EQU 08h
RECV_HIGH       EQU 00h
RECV_LOW        EQU 08h
ENDIF ;IFDEF PARPORT_LANC_INTERFACE

SET_DS MACRO

  push  ax
  mov   ax, cs:ds_reg
  .if (ax == 0)
    pop   ax
    mov   al, E__LANC_NOT_INITIALIZED
    ret
  .endif
  mov   ds, ax
  pop   ax

ENDM

PRG_DATA_SEGM SEGMENT PARA PUBLIC 'DATA' USE16

; program state variables
calibrated          byte 0
opened              byte 0

; LANC variables
lanc_channel        byte -1         ; -1=not specified yet, 0-2 on IRdeo for X1(=0), X2(=1) and X3(=2)

; serial port variables
com_port            byte -1         ; -1=not specified yet, 0=COM1, 1=COM2, 2=COM3, 3=COM4
IFDEF IRDEO_LANC_INTERFACE
ser_port_addr       word ?
ser_port_lcr        word ?
ser_port_mcr        word ?
ser_port_msr        word ?
ENDIF ;IFDEF IRDEO_LANC_INTERFACE

IFDEF PARPORT_LANC_INTERFACE
; parallel port variables
par_port_addr       word ?
par_port_statusreg  word ?
par_port_controlreg word ?
ENDIF ;IFDEF PARPORT_LANC_INTERFACE

; receive data variables
recv_data           byte 16 dup (?)
bytes_recvd         byte 0          ; bytes in recv_data - buffer
send_byte_recvd     byte 0          ; bytes received from the send_byte function

;calibration values
loops_per_bit       dword ?
loops_per_halfbit   dword ?

PRG_DATA_SEGM ENDS

CONST           SEGMENT PARA USE16 PUBLIC 'CODE'

com1_string     byte "/COM1", 0
com2_string     byte "/COM2", 0
com3_string     byte "/COM3", 0
com4_string     byte "/COM4", 0
x1_string       byte "/X1", 0
x2_string       byte "/X2", 0
x3_string       byte "/X3", 0

CONST           ENDS

CODEVAR         SEGMENT PARA USE16 PUBLIC 'CODE'

ds_reg          word 0

CODEVAR         ENDS

CODEGROUP       GROUP    CONST, CODEVAR, MYCODE

MYCODE SEGMENT PARA PUBLIC 'CODE' USE16
.386p

assume ds:PRG_DATA_SEGM

IODELAY MACRO
  out 0E1h, ax
ENDM

;in:    ecx=microseconds to wait
;out:   nothing
delay proc uses ax ecx

  .repeat
    .repeat                           ; wait for 1->0 transition
      in al, 61h
      IODELAY
    .until !(al & 00010000b)
    .repeat                           ; wait for 0->1 transition
      in al, 61h
      IODELAY
    .until (al & 00010000b)
    sub   ecx, 30
  .until (carry?)

  ret

delay endp


;IN:    BL          Value to compare with.
;       BH          Mask to apply to value read from port.
;       ECX         Amount of time to wait (in ticks at 14,318/12 MHz)
;       DX          Port to read from.
;OUT:   carry set   Timeout
;       carry clear Read OK
wait_port_ticks proc uses ax ecx

  .repeat
    in    al, dx
    and   al, bh
    .break .if (al == bl)
    .repeat                           ; wait for 1->0 transition
      in al, 61h
      IODELAY
    .until !(al & 00010000b)
    .repeat                           ; wait for 0->1 transition
      in al, 61h
      IODELAY
    .until (al & 00010000b)
    sub   ecx, 2*18
  .until (carry?)

  ret

wait_port_ticks endp


;in:    ecx     loops to perform
;out:   nothing
loop_delay proc uses ecx

  inc   ecx
  .repeat
    dec   ecx
  .until (ecx == 0)
  
  ret

loop_delay endp


;in:    nothing
;out:   nothing
init_comport proc uses ax dx

 ; get serial port address from com port number
  push  es
  push  bx
  mov   ax, 40h
  mov   es, ax

IFDEF IRDEO_LANC_INTERFACE
  movzx bx, com_port
  .if (bx > 3)                      ; must be 0-3 for COM1-COM4
    pop   bx
    pop   es
    ret
  .endif
  shl   bx, 1
  mov   ax, es:[bx]
  mov   ser_port_addr, ax
ENDIF ;IFDEF IRDEO_LANC_INTERFACE


; this is for the parallel port LANC (Pins 15,17,18)
IFDEF PARPORT_LANC_INTERFACE
  mov   bx, 8
  mov   ax, es:[bx]
  mov   par_port_addr, ax
  inc   ax
  mov   par_port_statusreg, ax
  inc   ax
  mov   par_port_controlreg, ax
ENDIF ;IFDEF PARPORT_LANC_INTERFACE

  pop   bx
  pop   es

IFDEF IRDEO_LANC_INTERFACE
  mov   ax, ser_port_addr
  mov   bx, LCR
  add   bx, ax
  mov   ser_port_lcr, bx
  mov   bx, MCR
  add   bx, ax
  mov   ser_port_mcr, bx
  mov   bx, MSR
  add   bx, ax
  mov   ser_port_msr, bx

  mov   dx, ser_port_mcr
  mov   al, 1           ; LANC-channel X0 on IRdeo
  out   dx, al
ENDIF ;IFDEF IRDEO_LANC_INTERFACE

  mov   dx, SEND_PORT
  mov   al, SEND_LOW
  out   dx, al

  ret

init_comport endp


;in:    al      byte to send
;out:   al      byte received
byte_out proc uses bx ecx dx si

  push  ax
  mov   ah, al

  xor   bx, bx
  mov   si, 9                       ; 8 bits per byte + stop bit
  mov   ecx, loops_per_halfbit

 ; wait for startbit to occur
  mov   dx, RECEIVE_PORT
  .repeat
    in    al, dx
    and   al, RECV_MASK
  .until (al == RECV_HIGH)

 ; send out bit per bit
  mov   dx, SEND_PORT
  .repeat
;;
;;ecx is still loaded:
;;
;;  mov   ecx, loops_per_halfbit
    call  loop_delay
   ; receive the bit
    mov   dx, RECEIVE_PORT
    in    al, dx
    and   al, RECV_MASK
    shr   bx, 1
    .if (al == RECV_HIGH)
      or    bx, 80h
    .endif
    call  loop_delay
    
    .if (ah & 1)
      mov   al, SEND_HIGH                  ; send logical high
    .else
      mov   al, SEND_LOW                   ; send logical low
    .endif
    mov   dx, SEND_PORT
    out   dx, al
    shr   ah, 1
    dec   si
  .until (si == 0)
  call  loop_delay

  pop   ax
  mov   al, bl

  ret

byte_out endp

comment ~
;in:    al      byte to send
;out:   nothing
byte_out proc uses ax ecx dx si

  mov   ah, al

  mov   si, 9                       ; 8 bits per byte + stop bit
  mov   ecx, loops_per_bit

 ; wait for startbit to occur
  mov   dx, RECEIVE_PORT
  .repeat
    in    al, dx
    and   al, RECV_MASK
  .until (al == RECV_HIGH)

 ; send out bit per bit
  mov   dx, SEND_PORT
  .repeat
;;
;;ecx is still loaded:
;;
;;  mov   ecx, loops_per_bit
    call  loop_delay
    .if (ah & 1)
      mov   al, SEND_HIGH                  ; send logical high
    .else
      mov   al, SEND_LOW                   ; send logical low
    .endif
    out   dx, al
    shr   ah, 1
    dec   si
  .until (si == 0)

  ret

byte_out endp
endcomment ~

;in:    nothing
;out:   al      received byte
;       ebx     time to start receiving
byte_in proc uses ecx dx si

 ; wait for startbit to occur
  xor   ebx, ebx
  mov   dx, RECEIVE_PORT
  .repeat
    in    al, dx
    and   al, RECV_MASK
    inc   ebx
  .until (al == RECV_HIGH)

  mov   ecx, loops_per_halfbit
  call  loop_delay

  mov   ecx, loops_per_bit

  mov   si, 8                       ; 8 bits per byte

 ; receive bit per bit
  xor   ah, ah
  mov   dx, RECEIVE_PORT
  .repeat
;;
;;ecx is still loaded:
;;
;;  mov   ecx, loops_per_bit
    call  loop_delay
    in    al, dx
    and   al, RECV_MASK
    shr   ah, 1
    .if (al == RECV_HIGH)
      or    ah, 80h
    .endif
    dec   si
  .until (si == 0)
  call  loop_delay                  ; wait for the stop bit to occur
  mov   al, ah

  ret

byte_in endp


GATE_LOW    EQU 0Ch
GATE_HIGH   EQU 0Dh
OP_MUL      EQU 0
OP_DIV      EQU 1
OP_ADD      EQU 0
OP_SUB      EQU 1


;in:    ax      ticks to calibrate the loop for
;out:   ecx     loops for given ticks
calibrate_loop proc uses ax bx edx
; register usage:
; ax=temp
; bl=operation for the corrector (div or mul)
; bh=operation for the loop counter (add or sub)
; ecx= loop counter
; edx= corrector

  pushf
  cli

  mov   bx, ax

 ; init timer
  mov   al, GATE_LOW
  out   61h, al
  mov   al, 10110010b   ; timer 2, write 16 bit value, mode 1, binary
  out   43h, al
  mov   ax, bx
  out   42h, al
  IODELAY
  mov   al, ah
  out   42h, al

  mov   ecx, 1024
  mov   edx, ecx
  shr   edx, 1
  mov   bl, OP_MUL
  .repeat

   ; trigger the timer...
    mov   al, GATE_HIGH
    out   61h, al

   ; ..and wait for OUT to go low
    .repeat
      in    al, 61h
    .until !(al & 20h)

   ; do the loops
    call  loop_delay

   ; did the timer count to 0?
    in    al, 61h
    .if (al & 20h)
      mov   bl, OP_DIV
      mov   bh, OP_SUB
    .else
      mov   bh, OP_ADD
    .endif
    .if (bl == OP_DIV)
      shr   edx, 1
    .else
      shl   edx, 1
    .endif
    .if (bh == OP_ADD)
      add   ecx, edx
    .else
      sub   ecx, edx
    .endif

   ; disable gate
    mov   al, GATE_LOW
    out   61h, al

  .until (edx == 1)
  popf

  ret

calibrate_loop endp

public send_calibrate
send_calibrate proc far uses ax ecx

  mov   ax, TICKS_PER_SEC/9600
  call  calibrate_loop
  mov   loops_per_bit, ecx

  mov   ax, TICKS_PER_SEC/(9600*2)
  call  calibrate_loop
  mov   loops_per_halfbit, ecx

  ret

send_calibrate endp


set_defaults proc uses ax

  mov   al, lanc_channel
  .if (al == -1)
    mov   lanc_channel, 0             ; = X1 on IRdeo
  .endif

  mov   al, com_port
  .if (al == -1)
;;    mov   com_port, 0                 ; COM1
    mov   com_port, 1                 ; COM2
  .endif

  ret

set_defaults endp


;IN:    DS:SI       Pointer to first string
;       ES:DI       Pointer to second string
;OUT:   Zero set    strings are equal
compare_strings proc uses ax bx cx si di

 ; get length of first string
  push  si
  xor   cx, cx
  xor   al, al
  .repeat
    cmp   al, ds:[si]
    .break .if (zero?)
  .untilcxz
  not   cx
  mov   bx, cx
  pop   si
  
 ; get length of second string
  push  di
  xor   cx, cx
  xor   al, al
  .repeat
    cmp   al, es:[di]
    .break .if (zero?)
  .untilcxz
  not   cx
  pop   di

  cmp   bx, cx
  .if (!zero?)
   ; string lengths are different, strings couldn't be the same
    ret
  .endif

  repe  cmpsb
  cmp   cx, 0

  ret

compare_strings endp

;=============================================================================
;                               public functions
;=============================================================================
 OPTION PROC:public

;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_init
;
;DESC:  
;
;IN:    Nothing
;OUT:   Nothing
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_init proc far uses ax

  mov   ax, PRG_DATA_SEGM
  mov   cs:ds_reg, ax

  ret

lanc_if_init endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_deinit
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_deinit proc far uses ds
  
  SET_DS

  mov   al, NOERROR

  ret

lanc_if_deinit endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_set_parameter
;
;DESC:  
;
;IN:    ES:BX       Pointer to a zero-terminated command line string
;OUT:   AL          Error code
;       BL          0: Parameter accepted
;                   1: Parameter not for us
;NOTE:  The command line may contain LANC_IF specific and other commands.
;       The lanc_if_init() function should ignore the other commands.
;
;       Command line parameters for the IRdeo interface:
;       /COM[1|2|3|4]       to select serial port, default is COM1
;                           e.g.: /COM1
;       /X[1|2|3]           to select IRdeo LANC port, default is X1
;-----------------------------------------------------------------------------
lanc_if_set_parameter proc far uses si di ds
local accepted:byte

  SET_DS

  mov   accepted, 0

  mov   di, bx
  push  cs
  pop   ds

  mov   si, offset com1_string
  call  compare_strings
  .if (zero?)
    mov   com_port, 0
    mov   accepted, 1
  .endif

  mov   si, offset com2_string
  call  compare_strings
  .if (zero?)
    mov   com_port, 1
    mov   accepted, 1
  .endif

  mov   si, offset com3_string
  call  compare_strings
  .if (zero?)
    mov   com_port, 2
    mov   accepted, 1
  .endif

  mov   si, offset com4_string
  call  compare_strings
  .if (zero?)
    mov   com_port, 3
    mov   accepted, 1
  .endif

  mov   si, offset x1_string
  call  compare_strings
  .if (zero?)
    mov   lanc_channel, 0
    mov   accepted, 1
  .endif

  mov   si, offset x2_string
  call  compare_strings
  .if (zero?)
    mov   lanc_channel, 1
    mov   accepted, 1
  .endif

  mov   si, offset x3_string
  call  compare_strings
  .if (zero?)
    mov   lanc_channel, 2
    mov   accepted, 1
  .endif

  .if (accepted)
    mov   bl, 0
  .else
    mov   bl, 1
  .endif

  mov   al, NOERROR

  ret

lanc_if_set_parameter endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_open
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_open proc far uses ds

  SET_DS

  call  set_defaults

  call  init_comport

  mov   al, lanc_channel
  call  lanc_if_set_channel

  mov   al, NOERROR

  ret

lanc_if_open endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_close
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_close proc far uses ds
  
  SET_DS

  sti

  mov   al, NOERROR

  ret

lanc_if_close endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_set_channel
;
;DESC:  
;
;IN:    AL = LANC Channel to use (0-2 on IRdeo)
;OUT:   AL          Error code
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_set_channel proc far uses cx dx ds

  SET_DS

IFDEF IRDEO_LANC_INTERFACE

  push  ax
  .if (al > 2)
    pop   ax
    mov   al, E__LANC_WRONG_CHANNEL
    ret
  .endif

  inc   al                          ; convert channel from 0-2 to 1-3
  mov   ah, al
  mov   dx, ser_port_mcr
  in    al, dx
  or    al, ah
  out   dx, al

  mov   ecx, 50000
  call  delay                       ; wait for channel to stabilize

  pop   ax

ENDIF ;IFDEF IRDEO_LANC_INTERFACE

  mov   al, NOERROR

  ret

lanc_if_set_channel endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_check_active
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;       BL          0: LANC bus seems to be active
;                   1: LANC bus inactive or data wrong
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_check_active proc far uses ecx dx ds
local result:byte

  SET_DS

  push  ax
  push  bx
  mov   result, 0

  mov   dx, RECEIVE_PORT
  mov   bh, RECV_MASK
  mov   ecx, TICKS_PER_SEC / 50

  mov   bl, RECV_HIGH
  call  wait_port_ticks
  .if (!carry?)
    mov   bl, RECV_LOW
    mov   ecx, TICKS_PER_SEC / 50
    call  wait_port_ticks
  .endif

  .if carry?
    mov   result, 1
  .endif

done:

  pop   bx
  mov   bl, result

  pop   ax
  mov   al, NOERROR

  ret

lanc_if_check_active endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_calibrate
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;NOTE:  
;-----------------------------------------------------------------------------
lanc_if_calibrate proc far uses si ds

  SET_DS

  mov   calibrated, 1

  call  send_calibrate
  .if (carry?)
    mov   al, E__LANC_CALIBRATION_SEND
  .endif

done:

  ret

lanc_if_calibrate endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_telegram_syncronize
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;-----------------------------------------------------------------------------
lanc_if_telegram_syncronize proc far uses ebx cx edx ds
local errorcode:byte

  SET_DS

  push  ax

 ; ensure interrupts are off
  pushf
  cli

  .if (!calibrated)
    mov   errorcode, E__LANC_NOT_CALIBRATED
    jmp   done
  .endif

 ; reset the byte receive counter
  mov   send_byte_recvd, 0
  mov   bytes_recvd, 0

  mov   cx, 17
  xor   edx, edx
  .repeat
    call  byte_in                 ; out: al=received byte, ebx=time to start receiving
    .if (ebx > edx)
      mov   edx, ebx
    .endif
  .untilcxz
  shr   edx, 2

  mov   cx, 9
  .repeat
    call  byte_in                 ; out: al=received byte, ebx=time to start receiving
    .break .if (ebx > edx)
  .untilcxz
  .if (cx == 0)
    mov   errorcode, E__LANC_TELEGRAMS
    jmp   done
  .endif
 ; now we have received the first byte of a telegram
  inc   bytes_recvd

  .repeat
    call  byte_in                   ; out: al=received byte, ebx=time to start receiving
    inc   bytes_recvd
  .until (bytes_recvd >= 8)
  mov   bytes_recvd, 0

  mov   errorcode, NOERROR

done:

 ; restore previous interrupt state
  popf

  pop   ax
  mov   al, errorcode

  ret

lanc_if_telegram_syncronize endp


;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_receive_telegram
;
;DESC:  
;
;IN:    Nothing
;OUT:   AL          Error code
;       ES:BX       pointer to telegram (only if no error occured)
;
;NOTE:  If no errors occured the last 8 received bytes are a telegram.
;-----------------------------------------------------------------------------
lanc_if_receive_telegram proc far uses ecx dx ds

  SET_DS

  mov   al, send_byte_recvd         ; maybe the lanc_if_send function did already receive some bytes of this telegram
  mov   send_byte_recvd, 0          ; reset the counter
  mov   bytes_recvd, al             ; error free received bytes
  push  ebx
  .repeat

    call  byte_in                   ; out: al=received byte, ebx=time to start receiving
    movzx bx, bytes_recvd
    mov   recv_data[bx], al

    inc   bytes_recvd

  .until (bytes_recvd >= 8)
  pop   ebx
  
  push  ds
  pop   es
  movzx bx, bytes_recvd
  sub   bx, 8
  add   bx, offset recv_data

done:

  mov   al, NOERROR

  ret

lanc_if_receive_telegram endp

;-----------------------------------------------------------------------------
;PUBLIC FUNCTION:  lanc_if_send
;
;DESC:  
;
;IN:    al          first byte to send
;       ah          second byte to send
;OUT:   AL          Error code
;
;NOTE:  
;-----------------------------------------------------------------------------
public lanc_if_send
lanc_if_send proc far uses dx ds
local errorcode:byte

  SET_DS

  push  ax
  .if (!calibrated)
    mov   errorcode, E__LANC_NOT_CALIBRATED
    jmp   done
  .endif

  call  byte_out
  mov   recv_data[0], al

  mov   al, ah
  call  byte_out
  mov   recv_data[1], al

  mov   send_byte_recvd, 2
  mov   errorcode, NOERROR

done:
  pop   ax
  mov   al, errorcode

  ret

lanc_if_send endp

MYCODE ENDS

END
