{****************************************************************************}
{ Program: MAIN.PAS                                                          }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ Remote control                                                             }
{ . IrDA device on parallel port                                             }
{ . IrDA device on ANY port                                                  }
{ . Pronto on serial port                                                    }
{                                                                            }
{ Possible calls:                                                            }
{    PRONTO 1                   Uses Pronto on serial port COM1 (default)    }
{    PRONTO x                   Uses Pronto on serial port COMx              }
{    PRONTO 9                   Uses Pronto on serial port COM9              }
{    PRONTO LPT                 Uses IrDA device LPT1                        }
{    PRONTO $37A $08 HIGH $379 $80 HIGH                                      }
{            (1) (2) (3)  (4)  (5) (6)                                       }
{                              Manual setting of output/input of IrDA device }
{           The manual setting must have 6 parameters. These are:            }
{           (1)        Output port                                           }
{           (2)        Output data                                           }
{                      The 'Output port' is first read and then only the     }
{                      'Output data' bits are changed to (in)active levels   }
{                      and then written back to 'Output port'                }
{           (3)        Active level (HIGH or LOW)                            }
{                      The output will usually also work with this set to the}
{                      incorrect level, although the infrared transmitter    }
{                      will then be active when it is supposed to be idle    }
{           (4)        Input port                                            }
{           (5)        Input data                                            }
{                      The 'Input port' is read and only the 'Input data'    }
{                      bits are checked for there (in)active levels          }
{           (6)        Active level (HIGH or LOW)                            }
{                      When this level is set incorrectly then recording     }
{                      starts immediately                                    }
{          Note that above example is identical to using LPT1 ....           }
{    other examples                                                          }
{       PRONTO $3FC $02 HIGH $3FE $10 HIGH                                   }
{                      Uses COM1 RTS as output and CTS as input              }
{                                                                            }
{ Parallel port control:                                                     }
{  An IrDa device (the HSDL-1001 from Hewlett-Packard or any simular device) }
{  is connected to the parallel port. This IrDA device is connected in the   }
{  following way:                                                            }
{   . Receiver input (pin 4) to parallel port BUSY (pin 11)                  }
{   . Transmitter output (pin 6) to parallel port SELECT (pin 17)            }
{     This is one of the few outputs which are LOW at rest (only during      }
{     startup this line is HIGH, but the BIOS/DOS sets this to LOW).         }
{                                                                            }
{ Note: We assume an AT (or higher) is used. An XT is not supported.         }
{                                                                            }
{ Note: It is NOT possible to send and receive at the same time with such an }
{       IrDa device because the output signal feeds back directly on it's    }
{       input. This is a hardware limitation.                                }
{----------------------------------------------------------------------------}
{ Version                                                                    }
{  Prior to version 3.00 there were two individual programs                  }
{    <RemoteIrDA> version 1.16                                               }
{    <Pronto>     version 2.00                                               }
{  3.00   20020721  - First release                                          }
{                   - Old code bug corrected: range checking incorrect       }
{                                             .. SizeOf .. -> .. High ..     }
{  3.01   20020810  - Added support for more devices                         }
{                   - No longer uses VxD: direct control in main code        }
{  3.02   20030322  - Added such that illegal parameter will use NO hardware }
{                   - By default no hardware used                            }
{                   - Check on incorrect 5000/5001/6000/6001 data added      }
{****************************************************************************}

{ Recorded file format (internal format):                                    }
{ First 16 bytes:  TT TT 00 00  CC CC 00 00  II II 00 00  UU UU 00 00        }
{ Were:                                                                      }
{   TT TT  Recording type (number of bytes per data unit)                    }
{          02 00 = word                                                      }
{          04 00 = longint/dword                                             }
{   CC CC  Cycles required for xxx instructions                              }
{   II II  Number of xxx instructions                                        }
{   UU UU  Cycle units in 0.1 us almost always 848                           }
{ Note: the two higher bytes of these are always set to 00 so they can be    }
{       read as either words or longints/dwords.                             }
{                                                                            }
{ All other data which follows is the recorded data which are just numbers.  }
{ These numbers indicate for how xxx operations a '0' or '1' was             }
{ The sequence is:  on off on off on off ...etc..                            }
{ Example (using words and all in hex!):                                     }
{  01 23 45 67 89 AB CD EF   -> 01 23 = first word $2301=8961  IOs active    }
{                            -> 45 67 = 2nd word   $6745=26437 IOs in-active }
{                            -> 89 AB = 3rd word   $AB89=43913 IOs active    }
{                            -> CD EF = 4th word   $EFCD=61389 IOs in-active }
{ A number of 234 would indicate that 234 instructions long a 0 or 1 was     }
{ detected. The actual time for this is 234 * ((CC/II)*1000)/UU              }
{ The very last data is always equal to the overflow count...                }

{ Pronto learned file format:                                                }
{ The format of a Pronto learned code is a series of 4-digit hexadecimal     }
{ numbers (when ASCII is used).                                              }
{ It is made up of a header and the data. The header:                        }
{    0000 CCCC SSSS RRRR                                                     }
{    . CCCC = carrier frequency      Frequency = 1000000/(N * .241246)       }
{                                    1000000 / (Frequency * 0.241246)        }
{    . SSSS = total number of data numbers following for SINGLE command      }
{    . RRRR = total number of data numbers following for REPEAT command      }
{ The data itself consist of a series of two numbers (each 4-digits).        }
{ This data indicates the ON time and the OFF time of the carrier frequency. }
{ If 'SSSS' is not zero then these codes are send.                           }
{ Tf 'RRRR' is not zero then these codes are send (and repeated xx times).   }

unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ClipBrd, StdCtrls, SyncObjs, ExtCtrls, Mask, ComCtrls, MMSystem;

const
  CLPT1 = $378;                        // Address LPT1

  CIoOperations   = 1000;              // Number of I/O operations to determine average timing
  CTimerUnits     = 838;               // 0.838 us
  CTimerUnit      = CTimerUnits * 1E-6; // Smallest timer unit

  // Settings for generating RC5/RC5X/RC6/RC6A code
  CRCFrequency   = 36000;              // Default carrier frequency
  CRCHalfbitTime = 889E-06;            // 889 us half bit time
  CRCDelay       = 69;                 // Default 69 ms delay after codes


  CTimerChannel0  = $40;               // Timer channel 0 register
  CTimerChannel1  = $41;               // Timer channel 1 register
  CTimerChannel2  = $42;               // Timer channel 2 register
  CTimerCommand   = $43;               // Timer mode/command register
  CTimer2Readback = $20;               // Readback bit of output of timer 2

  CPpiPortB       = $61;                // PPI port B
  CPpiPortC       = $62;                // PPI port C

  COverflowCount  = $400000;           // Overflow (timeout) detection
  
  CPrinterStatus  = $01;               // Status register offset
  CPrinterControl = $02;               // Control register offset
  CTransmitLine   = $08;               // Transmit = select = bit 3 control register
                                       // Note: during reset of PC signal is HIGH, it is typically set LOW after booting up (BIOS/DOS)
                                       // Note: this signal is inverted on the output
  CReceiveLine    = $80;               // Receive = busy status register

  CRecordSize = $FFF0;                 // Size of recording buffer -> must be even sized
                                       // Odd indexes : '0' timing (active)
                                       // Even indexes: '1' timing (in-active)


  CType       = 0;                     // Index 0: First index is type of data (sample length)
  CCycles     = 1;                     // Index 1: Cycles required for xxx operations
  COperations = 2;                     // Index 2: The number of operations which made up 'Cycles' (typically 1000)
  CCycleUnit  = 3;                     // Index 3: Cyle unit timing in 0.1 us (typically 838)

  CMessageStartRecording = WM_APP + 400;

const
  // Main states for communication thread.
  // Every main state can have intermediate states:
  // i.e. $100..$1FF all are part of <CMachineStateSuspend>
  // If an error is detected then the state will negated
  CMachineStateSuspend   = $100;                 // Idle/suspended
  CMachineStateTerminate = $200;                 // Terminate
  CMachineStateIrstart   = $300;                 // Irstart
  CMachineStateIrlearn   = $400;                 // Irlearn
  CMachineStateCancel    = $500;                 // Cancel
  CMachineStateReboot    = $600;                 // Reboot

type
  PWordArray = ^TWordArray;
  TWordArray = array[0..32767] of Word;
  PDWordArray = ^TDWordArray;
  TDWordArray = array[0..32767] of Dword;

  TCommunicationThread = class(TThread)
  protected
    FComPort          : Byte;                    // Communication port handle
    FMachineState     : Integer;                 // Current machine state
    FMachineStates    : TCriticalSection;        // Access lock
    FNextMachineState : Integer;                 // Next machine state (after completing current one)
    FProntoMaxDataSize: Integer;                 // Max size of Pronto data array (used for recording only)
    FProntoDataSize   : Integer;                 // Size of Pronto data array (recording/transmitting)
    FProntoDataArray  : PWordArray;              // The Pronto data (recording/transmitting)
    procedure Execute; override;
    function  GetState: Integer;
    procedure SetState(NextState: Integer);
  public
    constructor Create(CommunicationPort: Byte);
    destructor  Destroy; override;
    property    State: Integer              read GetState           write SetState;
    property    ProntoMaxDataSize: Integer  read FProntoMaxDataSize write FProntoMaxDataSize;
    property    ProntoDataSize: Integer     read FProntoDataSize    write FProntoDataSize;
    property    ProntoDataArray: PWordArray read FProntoDataArray   write FProntoDataArray;
  end;
  
  TfrmMain = class(TForm)
    btnExit: TButton;
    SaveDialog1: TSaveDialog;
    OpenDialog1: TOpenDialog;
    grpLoad: TGroupBox;
    btnLoad: TButton;
    btnLoadClipboard: TButton;
    grpSave: TGroupBox;
    btnSavePronto: TButton;
    grpRecord: TGroupBox;
    btnRecord: TButton;
    grpSend: TGroupBox;
    btnSendPronto: TButton;
    Panel1: TPanel;
    Label2: TLabel;
    Label1: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    rgrpGenerateRC: TRadioGroup;
    lblCustomerCode: TStaticText;
    lblSystem: TStaticText;
    lblCommand: TStaticText;
    lblData: TStaticText;
    mskCustomerCode: TMaskEdit;
    mskData: TMaskEdit;
    mskSystem: TMaskEdit;
    mskCommand: TMaskEdit;
    btnGenerateRC: TButton;
    grpRepeat: TGroupBox;
    chkRepeatRC: TCheckBox;
    chkRepeatPronto: TCheckBox;
    chkRepeat: TCheckBox;
    Label8: TLabel;
    tbTimeout: TTrackBar;
    lblTimeout: TLabel;
    chkToggle: TCheckBox;
    Label10: TLabel;
    StaticText1: TStaticText;
    mskFrequency: TMaskEdit;
    mskDelay: TMaskEdit;
    StaticText2: TStaticText;
    chkFrequencyFixate: TCheckBox;
    Label11: TLabel;
    Label12: TLabel;
    Label13: TLabel;
    Label14: TLabel;
    grpProntoFormat: TGroupBox;
    mmProntoData: TMemo;
    btnMemoToClipboard: TButton;
    Label15: TLabel;
    grpIrlearn: TGroupBox;
    chkAllowSuspiciousCode: TCheckBox;
    Label7: TLabel;
    Label9: TLabel;
    Label16: TLabel;
    mskRepeats: TMaskEdit;
    lblRepeats: TStaticText;
    Label17: TLabel;
    Label18: TLabel;
    procedure btnExitClick(Sender: TObject);
    procedure btnRecordClick(Sender: TObject);
    procedure btnSendClick(Sender: TObject);
    procedure btnSendProntoClick(Sender: TObject);
    procedure btnLoadClick(Sender: TObject);
    procedure btnLoadClipboardClick(Sender: TObject);
    procedure btnSaveInternalClick(Sender: TObject);
    procedure btnSaveProntoClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure rgrpGenerateRCClick(Sender: TObject);
    procedure btnGenerateRCClick(Sender: TObject);
    procedure btnMemoToClipboardClick(Sender: TObject);
    procedure mmProntoDataChange(Sender: TObject);
  private
    { Private declarations }
    FCommunicationThread: TCommunicationThread;
    { Internal format }
    procedure SendProntoFormat;
    procedure SendInternalFormat;
    function  LoadFile(UseClipboard: boolean): boolean;
    procedure WMRecord(var Message: TWMChar); message CMessageStartRecording;
    procedure ConvertToRepeatedCode;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;
  UsePronto : Boolean;
  UseLPTPort: Boolean;
  UseSpecial: Boolean;
  ConvertMessage : string;

  OutputPort   : Word;
  OutputAnd    : Byte;
  OutputOr     : Byte;
  OutputXor    : Byte;
  InputPort    : Word;
  InputAnd     : Byte;
  InputXor     : Byte;

  // Recording data
  RecordingDataArray: array[0..CRecordSize] of Dword;      // Recording data. First 16 bytes are the timing information
  RecordingDataSize : Word;                                // Index/Size of recorded data, points to free entry

  // Pronto code format
  ProntoDataSize    : Integer;                             // Index/Size of Pronto data array
  ProntoDataArray   : TWordArray;                          // The Pronto data

implementation

{$R *.DFM}


{------------------------------------------------------------------------------
  Params   : <CRCIn>   Old CRC
             <Data>    Data to add to checksum
  Return   : <Result>  New CRC

  Descript : Calculate new CRC (checksum). Some data requires that a CRC is
             send/received. This function calculates this checksum.
  Notes    : We do this realtime although lookup tables could be used too.
 ------------------------------------------------------------------------------}
function CRC(CRCIn: Word; Data: Byte): Word;
var
  Loop: Byte;
  LCRC: Word;
begin
  LCRC := CRCIn xor (Data shl 8);
  for Loop := 0 to 7 do
  begin
    if (LCRC and $8000)<>0 then LCRC := (LCRC shl 1) xor $1021
                           else LCRC := (LCRC shl 1);
  end;
  Result := LCRC;
end;


{------------------------------------------------------------------------------
  Params   : <CommunicationPort> Communication port to use
  Return   : -

  Descript : Create the thread which handles all serial communication to
             the Pronto.
  Notes    : -
 ------------------------------------------------------------------------------}
constructor TCommunicationThread.Create(CommunicationPort: Byte);
begin
  // Startup and setup
  FComPort := CommunicationPort;
  FNextMachineState  := CMachineStateSuspend;
  FMachineState      := CMachineStateSuspend;
  FMachineStates     := TCriticalSection.Create;
  FProntoDataArray   := nil;
  FProntoMaxDataSize := 0;
  FProntoDataSize    := 0;
  FreeOnTerminate    := True;
  inherited Create(False);
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Destruction of the thread. We have to release anything the <Create>
             acquired.
  Notes    : -
 ------------------------------------------------------------------------------}
destructor TCommunicationThread.Destroy;
begin
  inherited Destroy;
  FMachineStates.Free;
end;


{------------------------------------------------------------------------------
  Params   : <NextState>  State to execute when thread is ready for next one
  Return   : -

  Descript : Set next state to execute when thread is ready for it.
             If the thread is suspended then it will be started.
  Notes    : There is only one 'next' state, so an older one is written over.
 ------------------------------------------------------------------------------}
procedure TCommunicationThread.SetState(NextState: Integer);
begin
  FMachineStates.Acquire;
  if Suspended then
  begin
    FMachineState := NextState;
    FNextMachineState := CMachineStateSuspend;
    Resume;
  end
  else
    FNextMachineState := NextState;
  FMachineStates.Release;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Current state being executed

  Descript : Get current state.
  Notes    : A negative result or a <CMachineStateSuspend> indicates that the
             the thread is ready for a new state to execute.
 ------------------------------------------------------------------------------}
function TCommunicationThread.GetState: Integer;
begin
  Result := FMachineState;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : The thread which handles all serial communication to
             the Pronto.
  Notes    : The main application should use the <State> property for
             access to the thread.
 ------------------------------------------------------------------------------}
procedure TCommunicationThread.Execute;
var
  LComId    : THandle;                           // Communication handle
  LTimeOuts : TCOMMTIMEOUTS;                     // Communication timeouts
  LComDCB   : TDCB;                              // Communication Device Control Block }
  LNewState          : Integer;
  LTimer             : Longint;                  // Timer
  LReceivedData      : AnsiString;               // Received data
  LSendData          : AnsiString;               // Transmit data
  LReceivedCharacters: Dword;                    // Characters received
  LSendCharacters    : Dword;                    // Characters transmitted
  LResult            : Boolean;                  // Result from intermediate functions
  LCheckString       : AnsiString;               // String used for checking and other miscellanous operations
  LProntoDataIndex   : Integer;                  // Index into Pronto data array
  LRetries           : Integer;                  // Retry counter
  LError             : Integer;                  // Additiona error indication


  {------------------------------------------------------------------------------
    Params   : <Timer>   Timeout value to check upon
    Return   : <Result>  True if timed out

    Descript : Check if timed out
    Notes    : -
   ------------------------------------------------------------------------------}
  function TimerDone(Timer: Longint): Boolean;
  var
    Delta: Longint;
  begin
    Delta := Longint(TimeGetTime) - Timer;
    Result := (Delta >= 0) or (Delta < -1000000);          // The -1000000 for additional security (TimeGetTime overflow)
  end;


  {------------------------------------------------------------------------------
    Params   : <TimeOut>  Timeout in ms
    Return   : <Result>   Timeout value to be used with <TimerDone>

    Descript : Return timer inforation according to requested timeout
               to be used with <TimerDone>
    Notes    : -
   ------------------------------------------------------------------------------}
  function StartTimer(TimeOut: Longint): Longint;
  begin
    Result := Longint(TimeGetTime) + TimeOut;
  end;


  {------------------------------------------------------------------------------
    Params   : <DataSource>  Pointer to source learned code data
               <DataSize>    Size of learned code data
               <DataIndex>   Index into source data
    Return   : <Result>      Constructed data to transmit
                             Empty if error
               <DataIndex>   New index into source data

    Descript : Construct data to be transmitted (a single XModem data block)
               including checksum and block number.
    Notes    : -
   ------------------------------------------------------------------------------}
  function GenerateBlockData(DataSource: PWordArray; DataSize: Integer; var DataIndex: Integer): AnsiString;
  var
    OutString     : AnsiString;
    BlockChecksum : Word;
    BlockLoop     : Integer;
  begin
    OutString := '';
    if (DataSource = nil) or (DataSize = 0) then
      Exit;
    BlockChecksum := 0;
    // Set type of block and block number (both normal and inverted)
    OutString := #01;                                // Type of block == 128 bytes == 64 Pronto data
    OutString := OutString + chr((DataIndex div $40)+1);
    OutString := OutString + chr(((DataIndex div $40)+1) xor $FF);
    // A single block is always made up of 128 bytes == 64 words (1 word == 1 Pronto data)
    for BlockLoop := 0 to 63 do
    begin
      if DataIndex < DataSize then
      begin
        // We have valid data
        BlockChecksum := CRC(BlockChecksum, ord(DataSource[DataIndex] and $FF00 shr 8));
        BlockChecksum := CRC(BlockChecksum, ord(DataSource[DataIndex] and $FF));
        OutString := OutString + chr(ord(DataSource[DataIndex] and $FF00 shr 8));
        OutString := OutString + chr(ord(DataSource[DataIndex] and $FF));
      end
      else
      begin
        // No data available so use zeroes instead
        BlockChecksum := CRC(BlockChecksum, 0);
        BlockChecksum := CRC(BlockChecksum, 0);
        OutString := OutString + #0 + #0;
      end;
      inc(DataIndex);
    end;
    OutString := OutString + chr((BlockChecksum and $FF00) shr 8);
    OutString := OutString + chr(BlockChecksum and $FF);
    Result := OutString;
  end;


  {------------------------------------------------------------------------------
    Params   : <InString>    String data to process
               <DataSource>  Pointer to data
               <DataMaxSize> Max size of available space
               <DataSize>    Highest index used (+1)
    Return   : <Result>      False if no room (> <DataMaxSize>) or error in data
               <DataSize>    New higest index used

    Descript : Extract data block data from received data and check
               checksum and block number.
    Notes    : -
   ------------------------------------------------------------------------------}
  function ExtractBlockData(InString: AnsiString; DataSource: PWordArray; DataMaxSize: Integer; var DataSize: Integer): Boolean;
  var
    BlockChecksum : Word;
    BlockCheck    : Word;
    BlockLoop     : Integer;
    BlockSize     : Integer;
    DataIndex     : Integer;
  begin
    Result := False;
    if (DataSource = nil) or (DataSize > DataMaxSize) then
      Exit;
    if Length(InString) = 0 then                 // Not empty
      Exit;
    // Check blocklength and size of data
    case InString[1] of
      #$01: if Length(InString) <> (128+5) then
              Exit
            else
              BlockSize := 64;
      #$02: if Length(InString) <> (1024+5) then
              Exit
            else
              BlockSize := 512;
      else  Exit;
    end;
    // Check valid block number
    DataIndex := Ord(InString[2]);
    if Ord(InString[2]) <> (Ord(InString[3]) xor $FF) then
      Exit;
    if DataIndex = 0 then
      Exit;
    Dec(DataIndex);                              // Adjust block number
    DataIndex := DataIndex * BlockSize;          // Starting index in block
    BlockChecksum := 0;
    // A single block is always made up of 128/1024 bytes == 64/512 words (1 word == 1 Pronto data)
    for BlockLoop := 0 to BlockSize-1 do
    begin
      if DataIndex < DataMaxSize then            // Room available
      begin
        if (DataIndex+1) > DataSize then         // Adjust size of result data (+1 because size is not zero based..)
          DataSize := DataIndex + 1;
        // Add data to checksum
        BlockChecksum := CRC(BlockChecksum, ord(InString[(BlockLoop*2)+4]));
        BlockChecksum := CRC(BlockChecksum, ord(InString[(BlockLoop*2)+5]));
        BlockCheck := (ord(InString[(BlockLoop*2)+4]) shl 8) or ord(InString[(BlockLoop*2)+5]);
        DataSource[DataIndex] := BlockCheck;
      end
      else
        Exit;                                    // Exit if no room available
      inc(DataIndex);
    end;
    // Finally check the checksum
    BlockCheck := (ord(InString[(BlockSize*2)+4]) shl 8) or ord(InString[(BlockSize*2)+5]);
    if BlockCheck = BlockChecksum then
      Result := True;
  end;


  {------------------------------------------------------------------------------
    Params   : <CurrentState>  State to be handled
    Return   : <Result>        State to be handled in next run

    Descript : The IRSTART states.
    Notes    : -
   ------------------------------------------------------------------------------}
  function StateMachineIrstart(CurrentState: Integer): Integer;
  begin
    Result := CurrentState;
    case CurrentState of
      CMachineStateIrstart:                      // Initiate 'irstart': clear input buffer and send break
        begin
          if (FProntoDataArray = nil) or (FProntoDataSize = 0) then
            Result := -CurrentState
          else
          begin
            PurgeComm(LComId, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
            LResult := SetCommBreak(LComId);
            LTimer := StartTimer(10);
            if LResult then
              Result := CMachineStateIrstart + $01
            else
              Result := -CurrentState;             // Indicate error
          end;
        end;
      CMachineStateIrstart + $01:                // Wait for elapsed time for break signal
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateIrstart + $02;
        end;
      CMachineStateIrstart + $02:                // End break signal
        begin
          LResult := ClearCommBreak(LComId);
          if LResult then
            Result := CMachineStateIrstart + $03
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrstart + $03:                // Wait for '!' response
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            LTimer := StartTimer(10);
            if LCheckString = '!' then
              Result := CMachineStateIrstart + $04
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrstart + $04:                // Necessary delay after receiving '!'
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateIrstart + $05
        end;
      CMachineStateIrstart + $05:                // Send 'irstart'#13
        begin
          LSendData := 'irstart'#13;
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrstart + $06
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrstart + $06:                // Check for 'C' response
        begin
          LProntoDataIndex := 0;                 // Start of Pronto data
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            if LCheckString = 'C' then
              Result := CMachineStateIrstart + $07
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrstart + $07:                // Send a single block of data
        begin
          LSendData := GenerateBlockData(FProntoDataArray, FProntoDataSize, LProntoDataIndex);
          if length(LSendData) = 0 then
            Result := -CurrentState
          else
          begin
            LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
            if LResult then
              LResult := FlushFileBuffers(LComId);
            if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
              Result := CMachineStateIrstart + $08
            else
              Result := -CurrentState;             // Indicate error
          end;    
        end;
      CMachineStateIrstart + $08:                // Check for acknowledge of a send block of data
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            case LCheckString[1] of
              // Positive acknowledge: either send next block of data or start the ending part
              #$06:
                begin
                  if LProntoDataIndex < FProntoDataSize then
                    Result := CMachineStateIrstart + $07
                  else
                    Result := CMachineStateIrstart + $09;
                end;
              // For a negative acknowledge we need to retransmit the same block so correct our index
              #$15:
                begin
                  if LProntoDataIndex > 63 then
                    Dec(LProntoDataIndex, 64)
                  else
                    LProntoDataIndex := 0;
                  Result := CMachineStateIrstart + $07;
                end;
            else
              Result := -CurrentState;
            end;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrstart + $09:                // All blocks of data transmitted: now do the ending part
        begin
          LSendData := #$04;                     // End Of Text
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrstart + $0A
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrstart + $0A:                // Check for Negative Acknowledge response
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            if LCheckString[1] = #$15 then
              Result := CMachineStateIrstart + $0B
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrstart + $0B:                // After the EOT we send a NAK
        begin
          LSendData := #$04;                     // End Of Text
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrstart + $0C
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrstart + $0C:                // Check for Positive Acknowledge response
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          LTimer  := StartTimer(30);             // Delay needed before 'irstop' send
          if LResult and (LReceivedCharacters <> 0) then
          begin
            if LCheckString[1] = #$06 then
              Result := CMachineStateIrstart + $0D
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrstart + $0D:                // Necessary delay after receiving '!'
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateIrstart + $0E
        end;
      CMachineStateIrstart + $0E:                // Send 'irstop'#13
        begin
          LSendData := 'irstop'#13;
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          LTimer  := StartTimer(100);
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrstart + $0F
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrstart + $0F:                // Wait for timeout
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateIrstart + $FF
        end;
      // Next code is ONLY valid when Pronto is active so we DON'T use this
//      CMachineStateIrstart + $10:                // Wait for '!' response
//        begin
//          SetLength(LReceivedData, 10);
//          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
//          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
//          if LResult and (LReceivedCharacters <> 0) then
//          begin
//            if LCheckString = '!') then
//              Result := CMachineStateIrstart + $FF
//            else
//              Result := -CurrentState;
//          end
//          else
//            Result := -CurrentState;
//        end;
      CMachineStateIrstart + $FF:                // End of 'irstart'
        begin
          Result := CMachineStateSuspend;
        end;
    end;
  end;


  {------------------------------------------------------------------------------
    Params   : <CurrentState>  State to be handled
    Return   : <Result>        State to be handled in next run

    Descript : The IRLEARN states.
    Notes    : -
   ------------------------------------------------------------------------------}
  function StateMachineIrlearn(CurrentState: Integer): Integer;
  begin
    Result := CurrentState;
    case CurrentState of
      CMachineStateIrlearn:                      // Initiate 'irlearn': clear input buffer and send break
        begin
          if (FProntoDataArray = nil) or (FProntoMaxDataSize = 0) then
            Result := -CurrentState
          else
          begin
            PurgeComm(LComId, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
            LResult := SetCommBreak(LComId);
            LTimer := StartTimer(10);
            if LResult then
              Result := CMachineStateIrlearn + $01
            else
              Result := -CurrentState;             // Indicate error
          end;
        end;
      CMachineStateIrlearn + $01:                // Wait for elapsed time for break signal
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateIrlearn + $02;
        end;
      CMachineStateIrlearn + $02:                // End break signal
        begin
          LResult := ClearCommBreak(LComId);
          if LResult then
            Result := CMachineStateIrlearn + $03
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrlearn + $03:                // Wait for '!' response
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            LTimer := StartTimer(10);
            if LCheckString = '!' then
              Result := CMachineStateIrlearn + $04
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrlearn + $04:                // Necessary delay after receiving '!'
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateIrlearn + $05
        end;
      CMachineStateIrlearn + $05:                // Send 'irlearn 5000'#13
        begin
          LRetries := 11;                        // We try 11 queries
          LError   := 0;
          LSendData := 'irlearn 5000'#13;
          FProntoDataSize := 0;                  // Start of Pronto data
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrlearn + $06
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrlearn + $06:                // Check response
        begin
          SetLength(LReceivedData, 1048);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            if LReceivedCharacters <> 1 then
              Result := CMachineStateIrlearn + $07
            else
            begin
              // A single received character can (should) only indicate EOT
              case LCheckString[1] of
                #$04: Result := CMachineStateIrlearn + $09;
                else  Result := -CurrentState;
              end;
            end;
          end
          else
          begin
            // No result from Pronto (yet) so retry
            Dec(LRetries);
            if LRetries = 0 then
              Result := -CurrentState
            else
            begin
              LSendData := 'C';                  // Query with C
              LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
              if LResult then
                LResult := FlushFileBuffers(LComId);
              if (Longint(LSendCharacters) <> Length(LSendData)) or not LResult then
                Result := -CurrentState;         // Indicate error
            end;
          end;
        end;
      CMachineStateIrlearn + $07:                // We received a single block of data so process it
        begin
          if not ExtractBlockData(LCheckString, FProntoDataArray, FProntoMaxDataSize, FProntoDataSize) then
          begin
            LSendData := #$15;                   // NAK if something wrong
            LError    := -CurrentState;          // The Pronto continues as if nothing happened ...
          end
          else
            LSendData := #$06;                   // ACK if OK
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrlearn + $06
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrlearn + $09:                // All blocks of data received: now do the ending part
        begin
          LSendData := #$15;                     // NAK
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrlearn + $0A
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrlearn + $0A:                // Check for EOT
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            if LCheckString[1] = #$04 then
              Result := CMachineStateIrlearn + $0B
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateIrlearn + $0B:                // After the EOT we send a ACK
        begin
          LSendData := #$06;                     // ACK
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateIrlearn + $FF
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateIrlearn + $FF:                // End of 'irlearn'
        begin
          if (LError < 0) or (LRetries <= 6) then
          begin
            if LError < 0 then
              Result := LError
            else
              Result := -CurrentState;
          end
          else
            Result := CMachineStateSuspend;
        end;
    end;
  end;


  {------------------------------------------------------------------------------
    Params   : <CurrentState>  State to be handled
    Return   : <Result>        State to be handled in next run

    Descript : The IRCANCEL states.
    Notes    : -
   ------------------------------------------------------------------------------}
  function StateMachineCancel(CurrentState: Integer): Integer;
  begin
    Result := CurrentState;
    case CurrentState of
      CMachineStateCancel:                       // Initiate cancel
        begin
          LRetries := 32;
          PurgeComm(LComId, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
          with LTimeOuts do
          begin
            ReadIntervalTimeout         := 10;             // Timeout allowed between characters
            ReadTotalTimeoutMultiplier  := 0;
            ReadTotalTimeoutConstant    := 50;             // Small delay
            WriteTotalTimeoutMultiplier := 0;
            WriteTotalTimeoutConstant   := 0;
          end;
          // Now make the timeouts active
          SetCommTimeOuts(LComId, LTimeOuts);
          Result := CMachineStateCancel + $01;
        end;
      CMachineStateCancel + $01:                 // Send a single cancellation sequence
        begin
          LSendData := #$18#$18#$18#$18#$18;
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateCancel + $02
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateCancel + $02:                 // Check reply form Pronto
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            Result := CMachineStateCancel + $FF;
          end
          else
          begin
            Dec(LRetries);
            if LRetries = 0 then
              Result := -CurrentState
            else
              Result := CMachineStateCancel + $01;
          end;
        end;
      CMachineStateCancel + $FF:                 // End of cancellation
        begin
          with LTimeOuts do
          begin
            ReadIntervalTimeout         := 10;                 // Timoeut allowed between characters
            ReadTotalTimeoutMultiplier  := 0;                 
            ReadTotalTimeoutConstant    := 1000;               // Constant timeout
            WriteTotalTimeoutMultiplier := 0;
            WriteTotalTimeoutConstant   := 0;
          end;
          // Now make the timeouts active
          SetCommTimeOuts(LComId, LTimeOuts);
          Result := CMachineStateSuspend;
        end;
    end;
  end;


  {------------------------------------------------------------------------------
    Params   : <CurrentState>  State to be handled
    Return   : <Result>        State to be handled in next run

    Descript : The REBOOT states.
    Notes    : Only issues the command. It does not wait for it to complete.
   ------------------------------------------------------------------------------}
  function StateMachineReboot(CurrentState: Integer): Integer;
  begin
    Result := CurrentState;
    case CurrentState of
      CMachineStateReboot:                      // Initiate 'reboot': clear input buffer and send break
        begin
          PurgeComm(LComId, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
          LResult := SetCommBreak(LComId);
          LTimer := StartTimer(100);
          if LResult then
            Result := CMachineStateReboot + $01
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateReboot + $01:                // Wait for elapsed time for break signal
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateReboot + $02;
        end;
      CMachineStateReboot + $02:                // End break signal
        begin
          LResult := ClearCommBreak(LComId);
          if LResult then
            Result := CMachineStateReboot + $03
          else
            Result := -CurrentState;             // Indicate error
        end;
      CMachineStateReboot + $03:                // Wait for '!' response
        begin
          SetLength(LReceivedData, 10);
          LResult := ReadFile(LComId, LReceivedData[1], Length(LReceivedData), LReceivedCharacters, nil );
          LCheckString := Copy(LReceivedData, 1, LReceivedCharacters);
          if LResult and (LReceivedCharacters <> 0) then
          begin
            LTimer := StartTimer(10);
            if LCheckString = '!' then
              Result := CMachineStateReboot + $04
            else
              Result := -CurrentState;
          end
          else
            Result := -CurrentState;
        end;
      CMachineStateReboot + $04:                // Necessary delay after receiving '!'
        begin
          if TimerDone(LTimer) then
            Result := CMachineStateReboot + $05
        end;
      CMachineStateReboot + $05:                // Send 'reboot'#13
        begin
          LSendData := 'reboot'#13;
          LResult := WriteFile(LComId, LSendData[1], Length(LSendData), LSendCharacters, nil );
          if LResult then
            LResult := FlushFileBuffers(LComId);
          if (Longint(LSendCharacters) = Length(LSendData)) and LResult then
            Result := CMachineStateSuspend
          else
            Result := -CurrentState;             // Indicate error
        end;
    end;
  end;


  {------------------------------------------------------------------------------
    Params   : <CurrentState>  State to be handled
    Return   : <Result>        State to be handled in next run

    Descript : The handling of the different states.
    Notes    : -
   ------------------------------------------------------------------------------}
  function HandleStateMachine(CurrentState: Integer): Integer;
  begin
    Result := CurrentState;
    Result := StateMachineIrstart(Result);
    Result := StateMachineIrlearn(Result);
    Result := StateMachineCancel(Result);
    Result := StateMachineReboot(Result);
  end;

begin
  // First we setup the communication port
  if not (FComport in [1..9]) then                // Must be valid communication port
    Exit;
  LReceivedData := 'COM' + IntToStr(FComPort) + #0;
  // Try to open a handle to the communication port
  LComId := CreateFile(@LReceivedData[1],
    GENERIC_READ or GENERIC_WRITE,
    FILE_SHARE_READ or FILE_SHARE_WRITE,
    nil,
    OPEN_EXISTING,
    0,
    0);
  // If we were successful we still need to setup some stuff
  if LComId <> INVALID_HANDLE_VALUE then
  begin
    try
      // Get current settings
      GetCommState(LComId, LComDCB);
      // Setup baudrate, stopbits, parity and such
      with LComDCB do
      begin
        Baudrate := 115200;
        ByteSize := 8;
        Parity   := NOPARITY;
        StopBits := TWOSTOPBITS;
        Flags    := Flags or $0001;                           { Binary mode!! }
      end;
      // Make these changes active
      SetCommState(LComId, LComDCB);
      // We still need to set some timeouts (in our case no timeout at all), otherwise <ReadFile> keeps waiting
      with LTimeOuts do
      begin
        ReadIntervalTimeout         := 10;                 // Timeout allowed between characters
        ReadTotalTimeoutMultiplier  := 0;
        ReadTotalTimeoutConstant    := 1000;
        WriteTotalTimeoutMultiplier := 0;
        WriteTotalTimeoutConstant   := 0;
      end;
      // Now make the timeouts active
      SetCommTimeOuts(LComId, LTimeOuts);

      // The thread executes the same loop over and over until it is terminated.
      repeat
        FMachineStates.Acquire;
        LNewState := FMachineState;
        FMachineStates.Release;
        LNewState := HandleStateMachine(LNewState);        // Do state machine handling
        // Update the state machine variable and jump to next state if required
        // Since some variables can be accessed in parallel we have to handle them in a safe manner
        if (LNewState < 0) or                              // If an error
           (LNewState = CMachineStateSuspend) then         // or correct completion
        begin
          // If we are to go into suspension then we have to check if there is
          // a next request. If not we can actually go into suspension, otherwise
          // we go handling the next request.
          FMachineStates.Acquire;
          if FNextMachineState = CMachineStateSuspend then
          begin
            FMachineState := LNewState;
            FMachineStates.Release;
            Suspend;                                       // Wait for restart
          end
          else
          begin
            FMachineState := FNextMachineState;
            FNextMachineState := CMachineStateSuspend;
            FMachineStates.Release;
          end;
        end
        else
        begin
          FMachineStates.Acquire;
          FMachineState := LNewState;
          FMachineStates.Release;
        end;
        // Special if termination request
        FMachineStates.Acquire;
        if FMachineState = CMachineStateTerminate then
          Terminate;
        FMachineStates.Release;
      until Terminated;
    finally
      // When the thread is being terminated by any means we have to release the
      // communication port if we acquired it
      if LComId <> INVALID_HANDLE_VALUE then
        CloseHandle(LComId);
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Convert ASCII Pronto data into (global) word array data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure ConvertASCIItoPronto(InputString: AnsiString);
var
  Part : string;
  Value: Word;                                   // Conversion value
  Error: Integer;                                // Error indicator
begin
  // First convert some special codes (CR/LF/TAB) into spaces }
  ProntoDataSize := 0;
  while (Pos(#8 , InputString) <> 0) do InputString[Pos(#8 , InputString)] := ' ';
  while (Pos(#10, InputString) <> 0) do InputString[Pos(#10, InputString)] := ' ';
  while (Pos(#13, InputString) <> 0) do InputString[Pos(#13, InputString)] := ' ';
  while (length(InputString)<>0) do
  begin
    Part := '';
    // Remove leading spaces
    while (length(InputString)<>0) and (InputString[1]=' ') do
      Delete(InputString, 1, 1);
    // Get all characters until a space or end of the string
    while (length(InputString)<>0) and (InputString[1]<>' ') do
    begin
      Part := Part + InputString[1];
      Delete(InputString, 1, 1);
    end;
    // Convert the value
    if Part<>'' then
    begin
      Part := '$' + Part;
      Val(Part, Value, Error);
      if Error <> 0 then
      begin
        ProntoDataSize := 0;
        Exit;
      end
      else
      begin
        ProntoDataArray[ProntoDataSize] := Value;
        Inc(ProntoDataSize);
        if ProntoDataSize > High(ProntoDataArray) then
        begin
          ProntoDataSize := 0;
          Exit;
        end;
      end;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Result string
                       Empty if invalid

  Descript : Convert <ProntoDataArray> to string representation.
             Also corrects the <ProntoDataSize> (trailing zeroes and such)
  Notes    : -
 ------------------------------------------------------------------------------}
function ConvertProntoToASCII: AnsiString;
var
  AString: AnsiString;
  Loop   : integer;
begin
  ProntoDataSize := ((ProntoDataArray[2] + ProntoDataArray[3]) + 2) * 2;
  if (ProntoDataSize > 4) and
     (ProntoDataSize < High(ProntoDataArray)) then
  begin
    AString := '';
    for Loop := 0 to ProntoDataSize-1 do
      AString := AString + format('%.4x ', [ProntoDataArray[Loop]]);
    Result := AString;
  end
  else
    Result := '';
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Wait for a timeout. The timer has to be setup correctly.
             Do not use large timeouts because interrupts are disabled!
  Notes    : -
 ------------------------------------------------------------------------------}
procedure WaitForTimedOut; assembler;
asm
    pushf
    cli
    push  ax
    in    al, CPpiPortB
    test  al, CTimer2Readback
    jz    @wait1
  @wait0:
    in    al, CPpiPortB
    test  al, CTimer2Readback
    jnz   @wait0
    jmp   @end
  @wait1:
    in    al, CPpiPortB
    test  al, CTimer2Readback
    jz    @wait1
  @end:
    pop   ax
    popf
end;


{------------------------------------------------------------------------------
  Params   : <CtcClock>  Timer resolution
  Return   : -

  Descript : Setup timer resolution for the time out timer.
  Notes    : -
 ------------------------------------------------------------------------------}
procedure SetupTimedOutTimer(CtcClocks: byte);
begin                                    // Although a single ASM statement the BEGIN..END
                                         // is essential (otherwise 'CtcClcoks' won't be used!)
  asm
    pushf
    cli
    push  ax
    in    al, CPpiPortB
    and   al, $FD                        // Turn off speaker data
    or    al, $01                        // Turn on speaker gate
    out   CPpiPortB, al
    mov   al, $96                        // Timer 2, Lobyte only, mode 3, binary
    out   CTimerCommand, al
    mov   al, CtcClocks                  // Set resolution
    out   CTimerChannel2, al             // Start the timer
    pop   ax
    popf
  end;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Activate the IrDA output.
  Notes    : -
 ------------------------------------------------------------------------------}
procedure ActivateIrDAOutput;
asm
  push dx
  push ax
  mov  dx, [OutputPort]
  in   al, dx
  and  al, [OutputAnd]
  or   al, [OutputOr]
  xor  al, [OutputXor]
  out  dx, al
  pop  ax
  pop  dx
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Deactivate the IrDA output.
  Notes    : -
 ------------------------------------------------------------------------------}
procedure DeactivateIrDAOutput;
asm
  push dx
  push ax
  mov  dx, [OutputPort]
  in   al, dx
  and  al, [OutputAnd]
  xor  al, [OutputXor]
  out  dx, al                              { Output in-active }
  pop  ax
  pop  dx
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  True if active

  Descript : Check status of the IrDA input.
  Notes    : -
 ------------------------------------------------------------------------------}
function CheckIrDAInput: Boolean;
asm
  push dx
  push ax
  mov  dx, [InputPort]
  in   al, dx                              { Read input }
  and  al, [InputAnd]
  xor  al, [InputXor]
  mov  Result, False
  jz   @IsFalse
  mov  Result, True
@IsFalse:  
  pop  ax
  pop  dx
end;


{------------------------------------------------------------------------------
   Params  : <Size>       Size of buffer
             <Buffer>     Pointer to DWORD/LONGINT buffer
             <Timeout>    When to stop recording
   Returns : <Result>     TRUE if successful
             <Recorded>   Recorded data in buffer

   Descript: Receive IrDA from parallel port
   Notes   : Includes timeout
------------------------------------------------------------------------------}
function ReceiveIrDA(Size: Word; Buffer: PdwordArray; Timeout: Dword; var Recorded: Dword): boolean;
var
  Active        : Boolean;
  CheckActive   : Boolean;
  RecordingIndex: Word;
  StartTime     : Dword;
  EndTime       : Word;
  DeltaTime     : Word;
begin
  Result := False;

  StartTime := 0;
  EndTime   := 0;

  DeactivateIRDAOutput;                                    // Output in-active (remember that it is inverted on the output)

  // Determine timing of I/O operations
  asm
    pushf                                                  // Save current flags
    cli                                                    // Disable interrupts

    in    al, CPpiPortB                                    // Get port B
    and   al, $FD                                          // Turn off speaker data
    or    al, $01                                          // Turn on speaker gate
    out   CPpiPortB, al                                    // Write it back
    mov   al, $B6                                          // Timer 2, LoHi, mode 3
    out   CTimerCommand, al                                // Prepare timer
    mov   al, 0
    out   CTimerChannel2, al                               // Start the timer
    out   CTimerChannel2, al                               // Start the timer
    in    al, dx                                           // Allow some time to start
    in    al, dx
    in    al, dx
    in    al, dx
    in    al, dx
    in    al, dx

    mov   dx, [InputPort]
    mov   al, $80
    out   CTimerCommand, al                                // Latch timer 2
    in    al, CTimerChannel2                               // Read in latched timer 2
    mov   bl, al
    in    al, CTimerChannel2
    mov   bh, al
    mov   word ptr StartTime, bx

    mov   bx, CIOOperations                                // Go through it a number of times
@again:
    in    al, dx                                           // Read LPT port
    dec   bx
    jnz   @again

    mov   al, $80
    out   CTimerCommand, al                                // Latch timer 2
    in    al, CTimerChannel2                               // Read in latched timer 2
    mov   bl, al
    in    al, CTimerChannel2
    mov   bh, al

    mov   word ptr EndTime, bx                             // Save latched time into variables
    popf
  end;

  DeltaTime := abs(StartTime - EndTime) div 2;             // / 2 because mode 3 ticks twice as fast!
// FOR SOME REASON THERE IS A FACTOR 4 DIFFERENCE IN VXD TIMING IN WHAT WE WOULD EXPECT USING DOS
// THIS IS PROBABLY BECAUSE WINDOWS SETS THE TIMERS DIFFERENTLY.....
// THIS DOES NOT APPLY NORMAL CODE ...  DeltaTime := DeltaTime div 4;

  RecordingIndex := 0;
  Buffer[RecordingIndex] := 4;                             // 4 bytes/data
  inc(RecordingIndex);
  Buffer[RecordingIndex] := DeltaTime;                     // Deltatime
  inc(RecordingIndex);
  Buffer[RecordingIndex] := CIOOperations;                 // IO instructions used for Deltatime
  inc(RecordingIndex);
  Buffer[RecordingIndex] := CTimerUnits;                   // Units used for Deltatime
                                                           // NO increment of index because it is done in the next loop

  // Wait for the signal on the receive input becoming active
  StartTime := 1;
  repeat
    Active := CheckIRDAInput;                              // Check input
    inc(StartTime);
  until (Active or (StartTime >= COverflowCount));         // Wait for activation or timeout
  if (StartTime = COverflowCount) then                     // Timeout?
    Exit;

  while ((RecordingIndex < Size) and                       // A full recording indicates end
         (Buffer[RecordingIndex] <> Timeout)) do           // An overflow indicates end
  begin
    CheckActive := Active;                                 // Change to detect
    inc(RecordingIndex);
    Buffer[RecordingIndex] := 0;                           // Preset counter

    while ((Active = CheckActive) and                      // If not changed
           (Buffer[RecordingIndex] < Timeout)) do          // And no overflow
    begin
      Active := CheckIRDAInput;                            // Check input
      inc(Buffer[RecordingIndex]);                         // Increase counter
    end;
  end;

  Recorded := RecordingIndex;                             // Return size
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Data>       Data bit s to convert
            <FromBit>    Starting bit [0..15]
            <ToBit>      Ending bit to convert [0..15]
            <AZero>      How a '0' bit is to be converted to
            <AOne>       How a '1' bit is to be converted to
  Returns : <Result>     Result string
                                                                               
  Descript: Encode bits to a string representation
  Notes   : -
 ------------------------------------------------------------------------------}
function EncodeBits(Data: word; FromBit, ToBit: byte; AZero, AOne: string): string;
var
  BitString: string;
  Loop     : integer;
begin
  BitString := '';
  if not FromBit in [0..15] then Exit;
  if not ToBit   in [0..15] then Exit;

  if ToBit < FromBit then
  begin
    for Loop := FromBit downto ToBit do
    begin
      if (Data and (1 shl Loop)) = 0 then
        BitString := BitString + AZero
      else
        BitString := BitString + AOne;
    end;
  end
  else
  begin
    for Loop := FromBit to ToBit do
    begin
      if (Data and (1 shl Loop)) = 0 then
        BitString := BitString + AZero
      else
        BitString := BitString + AOne;
    end;
  end;
  Result := BitString;
end;


{------------------------------------------------------------------------------
  Params  : <BitStream>      Bitstream string to convert (made up of '1'/'0')
            <Delay>          Delay in ms (delay ending the code)
            <Halve>          Halve the timing information (RC6/RC6A)
            <UseRepeat>      Generate 'repeat' result
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Convert a bitstream into 1<>0 sequences (PRONTO format)
  Notes   : -
 ------------------------------------------------------------------------------}
function ZeroOneSequences(BitStream: string; Delay: word; Halve, UseRepeat: boolean): string;
var
  ProcessString: string;
  RCResult     : string;
  Count1       : integer;
  Count0       : integer;
  DelayUnit    : integer;
  Items        : integer;
  Frequency    : word;
  Frequency2   : word;
  CarrierUnit  : word;
  TimingUnit   : word;
  Error        : integer;
begin
  ProcessString := BitStream;
  Items         := 0;
  // We now want to know the 1->0 sequences.
  // The following sequences can occur:
  //  10    -> 1 unit  high, 1 unit  low
  //  100   -> 1 unit  high, 2 units low
  //  110   -> 2 units high, 1 unit  low
  //  1100  -> 2 units high, 2 units low

  // First the timing info to use
  val(frmMain.mskFrequency.EditText, Frequency, Error);
  if Error<>0 then Frequency := CRCFrequency;
  CarrierUnit := round(1E6/(Frequency  * 0.241246));       // Actual timing unit
  Frequency2  := trunc(1E6/(CarrierUnit * 0.241246));      // Actual frequency
  TimingUnit  := round(CRCHalfBitTime * Frequency2);       // -> 889 us / (1/Frequency)

  // First we have some minor cleaning up to do. We must make sure we start
  // with a '1' and end with a '1'. A leading '0' can be removed and a
  // trailing '0' might be added.
  // Because we do this our string is always correct for the next stages.
  if (length(ProcessString)<>0) and (ProcessString[1]='0') then
    Delete(ProcessString, 1, 1);
  if (length(ProcessString)<>0) and (ProcessString[length(ProcessString)]='1') then
    ProcessString := ProcessString + '0';
  // Now count them and while we are at it translate it into the result:
  // Each 1->0 sequence is translated into two (4-digit hexadecimal) numbers.
  // These two numbers indicate the ON time (1) and the OFF time (0) of the
  // sequence. It is in units of the carrier frequency. We already calculated/
  // defined the TimingUnits required for each bit in the string.
  // For the delay time we need to calculate it seperately.
  DelayUnit := round((Delay / 1000) * Frequency2);         // Delay in ms
  if DelayUnit < TimingUnit then
    DelayUnit := TimingUnit;
  while length(ProcessString)<>0 do
  begin
    inc(Items);
    Count0 := 0;
    Count1 := 0;
    while (length(ProcessString)<>0) and (ProcessString[1]='1') do   // The added security (length check) is not really necessary
    begin
      inc(Count1);
      delete(ProcessString, 1, 1);
    end;
    while (length(ProcessString)<>0) and (ProcessString[1]='0') do   // The added security (length check) is not really necessary
    begin
      inc(Count0);
      delete(ProcessString, 1, 1);
    end;
    // We now have counted the number of ones and zeroes in one sequence, now
    // convert them into hexadecimal numbers.
    // A special case is if all data is processed; then we need to use the
    // requested delay.
    if length(ProcessString)<>0 then
    begin
      if Halve then                                        // RC6/6A uses a halved timebase
        RCResult := RCResult + format(' %.4x %.4x', [(Count1 * TimingUnit) div 2, (Count0 * TimingUnit) div 2])
      else
        RCResult := RCResult + format(' %.4x %.4x', [Count1 * TimingUnit, Count0 * TimingUnit])
    end
    else
      if Halve then
        RCResult := RCResult + format(' %.4x %.4x', [(Count1 * TimingUnit) div 2, DelayUnit])
      else
        RCResult := RCResult + format(' %.4x %.4x', [Count1 * TimingUnit, DelayUnit]);
  end;
  { Now all we have to do is remove the single leading space }
  if (length(RCResult)<>0) and (RCResult[1]=' ') then
    Delete(RCResult, 1, 1);

//  RCResult := RCResult + ' ';
  { Now we have all our codes ina single string we only need to add the }
  { header:                                                             }
  { 0000 CCCC IIII 0000                                                 }
  { CCCC = carrier frequency      Frequency = 1000000/(N * .241246)     }
  {                               1000000 / (Frequency * 0.241246)      }
  { IIII = total items                                                  }
  {                         OR FOR REPEAT                               }
  { 0000 CCCC 0000 IIII                                                 }
  { CCCC = carrier frequency      Frequency = 1000000/(N * .241246)     }
  {                               1000000 / (Frequency * 0.241246)      }
  { IIII = total items                                                  }
  { Again all values are 4-digits hexadecimal                           }
  if UseRepeat then
    RCResult := '0000 ' + format('%.4x ', [CarrierUnit]) +
                format('0000 %.4x ', [Items]) + RCResult
  else
    RCResult := '0000 ' + format('%.4x ', [CarrierUnit]) +
                format('%.4x 0000 ', [Items]) + RCResult;

  Result := RCResult;
end;


{------------------------------------------------------------------------------
  Params  : <Toggle>         Toggle bit on or off
            <SystemAddress>  System address (0-31)
            <Command>        Command (0-127)
            <Delay>          Delay in ms
            <UseRepeat>      Generate repeat code
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC5 code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC5(Toggle: boolean; SystemAddress, Command, Delay: word; UseRepeat: boolean): string;
var
  RC5String: string;
begin
  Result    := '';
  if SystemAddress > 31   then Exit;
  if Command       > 127  then Exit;
  if Delay         > 5000 then Exit;

  // First we translate the system/command/toggle and such in straight binary
  // Format of a RC5 sequence
  //
  // Bitstream:
  // ss T SSSSS CCCCCC
  //
  //  ss = 10   Add 64 to command
  //  ss = 11   Use command as it is
  //  T         Toggle bit
  //  SSSSS     System bits (5)
  //  CCCCCC    Command bits (6)
  //
  //  ALL bits are biphase encoded:
  //  0 -> 10
  //  1 -> 01
  RC5String := '';
  if Command > 63 then RC5String := RC5String + EncodeBits(2, 1, 0, '10', '01')
                  else RC5String := RC5String + EncodeBits(3, 1, 0, '10', '01');
  if Toggle       then RC5String := RC5String + EncodeBits(1, 0, 0, '10', '01')
                  else RC5String := RC5String + EncodeBits(0, 0, 0, '10', '01');
  RC5String := RC5String + EncodeBits(SystemAddress, 4, 0, '10', '01');
  RC5String := RC5String + EncodeBits(Command, 5, 0, '10', '01');

  Result := ZeroOneSequences(RC5String, Delay, False, UseRepeat);
end;


{------------------------------------------------------------------------------
  Params  : <Toggle>         Toggle bit on or off
            <SystemAddress>  System address (0-31)
            <Command>        Command (0-127)
            <Data>           Command (0-63)
            <Delay>          Delay in ms
            <UseRepeat>      Generate repeat code
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC5X code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC5X(Toggle: boolean; SystemAddress, Command, Data, Delay: word; UseRepeat: boolean): string;
var
  RC5XString: string;
begin
  Result    := '';
  if SystemAddress > 31   then Exit;
  if Command       > 127  then Exit;
  if Data          > 63   then Exit;
  if Delay         > 5000 then Exit;

  // First we translate the system/command/toggle and such in straight binary
  // Format of a RC5X sequence
  // Bitstream:
  //  ss T SSSSS dddd CCCCCC DDDDDD
  //
  //  ss = 10   Add 64 to command      (To be biphase encoded)
  //  ss = 11   Use command as it is   (To be biphase encoded)
  //  T         Toggle bit             (To be biphase encoded)
  //  SSSSS     System bits (5)        (To be biphase encoded)
  //  dddd      Divider bits (4)       (NOT to be biphase encoded)
  //            '0000'
  //  CCCCCC    Command bits (6)       (To be biphase encoded)
  //  DDDDDD    Data bits (6)          (To be biphase encoded)
  //
  //  ALL bits are biphase encoded, except for the dddd bits:
  //  0 -> 10
  //  1 -> 01
  RC5XString := '';
  if Command > 63 then RC5XString := RC5XString + EncodeBits(2, 1, 0, '10', '01')
                  else RC5XString := RC5XString + EncodeBits(3, 1, 0, '10', '01');
  if Toggle       then RC5XString := RC5XString + EncodeBits(1, 0, 0, '10', '01')
                  else RC5XString := RC5XString + EncodeBits(0, 0, 0, '10', '01');
  RC5XString := RC5XString + EncodeBits(SystemAddress, 4, 0, '10', '01');
  RC5XString := RC5XString + '0000';
  RC5XString := RC5XString + EncodeBits(Command, 5, 0, '10', '01');
  RC5XString := RC5XString + EncodeBits(Data,    5, 0, '10', '01');

  Result := ZeroOneSequences(RC5XString, Delay, False, UseRepeat);
end;


{------------------------------------------------------------------------------
  Params  : <Toggle>         Toggle bit on or off
            <SystemAddress>  System address (0-255)
            <Command>        Command (0-255)
            <Delay>          Delay in ms
            <UseRepeat>      Generate repeat code
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC5X code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC6(Toggle: boolean; SystemAddress, Command, Delay: word; UseRepeat: boolean): string;
var
  RC6String: string;
begin
  Result    := '';
  if SystemAddress > 255   then Exit;
  if Command       > 255  then Exit;
  if Delay         > 5000 then Exit;

  // First we translate the system/command/toggle and such in straight binary
  // Format of a RC6 sequence
  // Bitstream:
  // hhhhhhhh TT SSSSSSSS CCCCCCCC
  //
  // hhhhhhhh  Header data (16)
  //            '1111110010010101' (NOT to be biphase encoded!)
  //  TT        Toggle (4)
  //            '0011' or          (NOT to be biphase encoded!)
  //            '1100'             (NOT to be biphase encoded!)
  //  SSSSSSSS  System bits  (8)   (To be biphase encoded)
  //  CCCCCCCC  Command bits (8)   (To be biphase encoded)
  //
  //  ONLY system and command bits are biphase encoded:
  //  0 -> 01
  //  1 -> 10               Note the inverted nature in comparison with RC5
  RC6String := '';
  RC6String := RC6String + '1111110010010101';
  if Toggle then RC6String := RC6String + '1100'
            else RC6String := RC6String + '0011';
  RC6String := RC6String + EncodeBits(SystemAddress, 7, 0, '01', '10');
  RC6String := RC6String + EncodeBits(Command,       7, 0, '01', '10');

  Result := ZeroOneSequences(RC6String, Delay, True, UseRepeat);
end;


{------------------------------------------------------------------------------
  Params  : <Toggle>         Toggle bit on or off
            <CustumerCode>   Customer code: 0..127 or 32768..65535
            <SystemAddress>  System address (0-255)
            <Command>        Command (0-255)
            <Delay>          Delay in ms
            <UseRepeat>      Generate repeat code
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC5X code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC6A(Toggle: boolean; CustomerCode,SystemAddress, Command, Delay: word; UseRepeat: boolean): string;
var
  RC6AString: string;
begin
  Result    := '';
  if ((CustomerCode  > 127) and (CustomerCode < 32768)) then Exit;
  if SystemAddress > 255   then Exit;
  if Command       > 255  then Exit;
  if Delay         > 5000 then Exit;

  // First we translate the system/command/toggle and such in straight binary
  // Format of a RC6A sequence
  //  Bitstream:
  //  hhhhhhhh TT s UUUUUUU SSSSSSSS CCCCCCCC
  //  hhhhhhhh TT s UUUUUUUUUUUUUUU SSSSSSSS CCCCCCCC
  //
  //  hhhhhhhh  Header data (17)
  //            '11111110010101001'  (NOT to be biphase encoded!)
  //  TT        Toggle (2/4)
  //            '0011' or            (NOT to be biphase encoded!)
  //            '1100'               (NOT to be biphase encoded!)
  //  s = 0     Custumer code 0..127       (To be biphase encoded)
  //  s = 1     Custumer code 32768..65536 (To be biphase encoded)
  //            Add 32768 to customer code
  //  UUUUUUU   Customer Code (7/15)       (To be biphase encoded)
  //            These are either 7 or 15 bits depending on the 's' size bit
  //  SSSSSSSS  System bits  (8)           (To be biphase encoded)
  //  CCCCCCCC  Command bits (8)           (To be biphase encoded)
  //
  //  ONLY customer code, system and command bits are biphase encoded:
  //  0 -> 01
  //  1 -> 10               Note the inverted nature in comparison with RC5
  RC6AString := '';
  RC6AString := RC6AString + '11111110010101001';
  if Toggle then RC6AString := RC6AString + '1100'
            else RC6AString := RC6AString + '0011';
  if CustomerCode > 127 then
  begin
    RC6AString := RC6AString + EncodeBits(1, 0, 0, '01', '10');
    RC6AString := RC6AString + EncodeBits(CustomerCode, 14, 0, '01', '10');
  end
  else
  begin
    RC6AString := RC6AString + EncodeBits(0, 0, 0, '01', '10');
    RC6AString := RC6AString + EncodeBits(CustomerCode, 6, 0, '01', '10');
  end;
  RC6AString := RC6AString + EncodeBits(SystemAddress, 7, 0, '01', '10');
  RC6AString := RC6AString + EncodeBits(Command,       7, 0, '01', '10');

  Result := ZeroOneSequences(RC6AString, Delay, True, UseRepeat);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>         Carrier frequency in Hz
                             0 if an error was detected

  Descript: Scan data for carrier data.
  Notes   : -
 ------------------------------------------------------------------------------}
function DetectCarrierFromRecording: Longint;
var
  Loop           : Integer;
  Calc           : Extended;
  Mult           : Extended;
  ActiveSample   : Longint;
  InActiveSample : Longint;
  Average        : Extended;
  Averaged       : Word;
begin
  Result := 0;
  // Sanity checks
  if RecordingDataSize = 0 then
    Exit;
  if (RecordingDataSize <= CCycleUnit) then
  begin
    RecordingDataSize := 0;
    Exit;
  end;
  if not(RecordingDataArray[CType] in [2,4]) then
  begin
    RecordingDataSize := 0;
    Exit;
  end;
  Mult := (RecordingDataArray[CCycles]/RecordingDataArray[COperations] * RecordingDataArray[CCycleUnit]) / 1000;
  // Go through all the data to detect carrier frequency
  // Depending on the source we have to go through the data differently
  Loop     := 4;                                 // Skip header
  Average  := 0;
  Averaged := 0;
  repeat
    if RecordingDataArray[CType] = 2 then                 // If words
    begin
      // Since our data is stored as longint we have to take them a part
      ActiveSample   := RecordingDataArray[Loop] and $FFFF;
      InActiveSample := (RecordingDataArray[Loop] shr 16) and $FFFF;
      inc(Loop);
    end
    else
    begin
      ActiveSample   := RecordingDataArray[Loop];
      inc(Loop);
      if Loop >= RecordingDataSize then
      begin
        RecordingDataSize := 0;
        Exit;
      end;
      InActiveSample := RecordingDataArray[Loop];
      inc(Loop);
    end;
    // Only takes samples which are no more than 1::4
    if ((ActiveSample <> 0) and (InActiveSample <> 0)) then
      if ((ActiveSample <= InActiveSample) and ((ActiveSample   / InActiveSample) > 0.25)) or
         ((ActiveSample >= InActiveSample) and ((InActiveSample / ActiveSample  ) > 0.25)) then
      begin
        Calc := (ActiveSample + InActiveSample) * Mult;
        Calc := Calc * 1E-6;
        Calc := 1/Calc;
        Average := Average + Calc;
        inc(Averaged);
      end;
  until Loop >= RecordingDataSize;
  if Averaged <> 0 then
    Result := Round(Average/Averaged);
end;


{------------------------------------------------------------------------------
  Params  : <GenerateRepeat>  True if a repeat code is to be generated
                              False if no repeat code is to be generated
  Returns : <Result>          1      Suspicious timing
                              0      No obvious errors
                              -1     No data
                              -2     Invalid data
                              -3     Invalid type
                              -4     No carrier detected
                              -5     Invalid number of data
                              -6     On/Off time, not carrier based signal
                              -7     Too much data
                              -8     Too much deviation from carrier signal  
            <ExitMessage>     Result message

  Descript: Convert internal format to Pronto format
            The direct data recording is changed into Pronto format.
            This takes the following into account:
            . the carrier frequency is determined
            . an ON signal (which is made up of the carrier frequency) is
              determined -> note that the carrier signal has most likely no
              equal on/off timing (which would be the theoretical input
            . ON and OFF signals are determined (carrier or not)
  Notes   : -
 ------------------------------------------------------------------------------}
function ConvertRecordingDataToProntoFormat(GenerateRepeat: boolean; var ExitMessage: string): Integer;
var
  Carrier      : Longint;
  Calc         : Extended;
  SourceTimeUs : Extended;
  TargetTimeUs : Extended;
  Loop         : Integer;
  OnTime       : Extended;
  OffTime      : Extended;
  CarrierCount : Integer;
  TooShort     : Boolean;
begin
  TooShort := False;
  // Do some sanity checks
  Result   := -1;
  ExitMessage := 'No data';
  if RecordingDataSize = 0 then
    Exit;
  Result   := -2;
  ExitMessage := 'Data contains invalid data';
  if RecordingDataSize < 6 then
    Exit;
  Result   := -3;
  if not(RecordingDataArray[CType] in [2,4]) then
    Exit;
  Result   := -4;
  ExitMessage := 'Invalid data (could not detect carrier)';
  Carrier := DetectCarrierFromRecording;
  if Carrier = 0 then Exit;

  // The carrier is usually a number rounded to 1000 Hz...
  Carrier := ((Carrier + 500) div 1000) * 1000;
  ExitMessage := format('Carrier: %d', [Carrier]);

  // To recap; the Pronto format is made up of a header and the data.
  //  The header:
  //    0000 CCCC IIII 0000
  //    . CCCC = carrier frequency      Frequency = 1000000/(N * .241246)
  //                                    N = 1000000 / (Frequency * 0.241246)
  //    . IIII = total number of data numbers following

  // Convert detected carrier into Pronto value
  Calc := 1E6 / (Carrier * 0.241246);
  ProntoDataArray[0] := $0000;
  ProntoDataArray[1] := Round(Calc);
  ProntoDataArray[2] := $0000;
  ProntoDataArray[3] := $0000;
  ProntoDataSize     := 4;

  SourceTimeUs := (((RecordingDataArray[CCycles] / RecordingDataArray[COperations]) * (RecordingDataArray[CCycleUnit]/1000)));
  TargetTimeUs := (1/Carrier)/2 * 1E6;           // Factor /2 is because 1 frequency cycle = on + off

  CarrierCount := 0;
  Loop := CCycleUnit+1;
  while Loop < RecordingDataSize do
  begin
    if RecordingDataArray[CType]=2 then                   // If word source
    begin
      OnTime  := RecordingDataArray[Loop] and $FFFF;
      OffTime := (RecordingDataArray[Loop] shr 16) and $FFFF;
      inc(Loop);
    end
    else
    begin                                        // If longint source
      OnTime := RecordingDataArray[Loop];
      inc(Loop);
      if Loop >= RecordingDataSize then
      begin
        Result := -5;
        ExitMessage := 'Invalid number of data';
        ProntoDataSize := 0;
        RecordingDataSize := 0;
        Exit;
      end;
      OffTime := RecordingDataArray[Loop];
      inc(Loop);
    end;
    OnTime  := OnTime  * SourceTimeUs;
    OffTime := OffTime * SourceTimeUs;
    if OnTime > (OffTime * 4) then
    begin
      Result := -6;
      ExitMessage := 'Invalid data detected (on-time >> off-time, not carrier based signal)';
      RecordingDataSize := 0;
      ProntoDataSize := 0;
      Exit;
    end;

    // A long off-time follows (end of carrier)
    if OffTime > (OnTime * 4) then
    begin
      inc(CarrierCount);
      ProntoDataArray[ProntoDataSize] := CarrierCount;
      if ProntoDataArray[ProntoDataSize] <= 4 then       // If less than 5 carrier cycles detected, probbaly error
        TooShort := True;
      inc(ProntoDataSize);
      if ProntoDataSize > High(ProntoDataArray) then
      begin
        Result := -7;
        ExitMessage := 'Too much data';
        RecordingDataSize := 0;
        ProntoDataSize := 0;
        Exit;
      end;
      // Convert off-time to equivalent of carrier cycles (=on+off)
      ProntoDataArray[ProntoDataSize] := round((OffTime - OnTime) / (TargetTimeUs * 2));
      if ProntoDataArray[ProntoDataSize] <= 4 then       // If less than 5 carrier cycles detected, probbaly error
        TooShort := True;
      inc(ProntoDataSize);
      if ProntoDataSize > High(ProntoDataArray) then
      begin
        Result := -7;
        ExitMessage := 'Too much data';
        RecordingDataSize := 0;
        ProntoDataSize := 0;
        Exit;
      end;
      CarrierCount := 0;
    end
    else                                         // If carrier signal
    begin
      if (OnTime + OffTime) > ((TargetTimeUs * 2) * 4 {1.5}) then
      begin
        Result := -8;
        ExitMessage    := format('Invalid data detected (too much deviation from carrier signal @ %d', [Loop]);
        RecordingDataSize := 0;
        ProntoDataSize := 0;
        Exit;
      end
      else inc(CarrierCount);
    end;
  end;
  if GenerateRepeat then
    ProntoDataArray[3] := (ProntoDataSize - 4) div 2
  else
    ProntoDataArray[2] := (ProntoDataSize - 4) div 2;
  if TooShort then
    ExitMessage := 'Conversion done (suspicious timing detected)'
  else
    ExitMessage := 'Conversion done';
  if TooShort then
    Result := +1
  else
    Result := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Convert into a single code with the sequence repeated if selected.
  Notes   : Converting into a repeated sequence is necessary when the Pronto is
            directly controlled because it does not generate a repeated code
            by means of it 'irstart' command.
            No checks done on valid data; it is assumed to be correct.
            Also no check on size is done
 ------------------------------------------------------------------------------}
procedure TfrmMain.ConvertToRepeatedCode;
var
  Repeats    : Integer;
  Loop       : Integer;
  Items      : Integer;
  Error      : Integer;
  RepeatCodes: Integer;
begin
  // Do not do it if it is a single code format
  if ProntoDataArray[2] <> 0 then
    Exit;
  Val(mskRepeats.EditText, RepeatCodes, Error);
  if Error <> 0 then
    Exit;
  if RepeatCodes = 0 then
    Exit;
  Inc(RepeatCodes);
  Items := ProntoDataArray[3];
  // Check the available size roughly
  if (ProntoDataSize > 4) and
     ((ProntoDataSize * RepeatCodes) < High(ProntoDataArray)) then
  begin
    for Repeats := 1 to RepeatCodes do
      for Loop := 0 to Items-1 do
      begin
        ProntoDataArray[4 + (Repeats * Items * 2) + (Loop * 2)] := ProntoDataArray[4 + (Loop * 2)];
        ProntoDataArray[5 + (Repeats * Items * 2) + (Loop * 2)] := ProntoDataArray[5 + (Loop * 2)];
      end;
    // Update sizes/length  
    ProntoDataArray[3] := 0;
    ProntoDataArray[2] := Items * RepeatCodes;
    ProntoDataSize := ((Items * RepeatCodes) * 2) + 4;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Message>
  Returns : -

  Descript: Recording button activated
  Notes   : -
 ------------------------------------------------------------------------------}
procedure TfrmMain.WMRecord(var Message: TWMChar);
begin
  btnRecordClick(nil);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Record IrDA data from parallel port to file (internal format)
  Notes   : Uses VxD recording although directly coded with direct I/O does work
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnRecordClick(Sender: TObject);
var
  BytesReturned  : Dword;
  InputString    : AnsiString;
  TheClipboard   : TClipboard;
  ConvertResult  : Integer;
begin
  if UsePronto then
  begin
    if FCommunicationThread = nil then
      ShowMessage('No recording source (Pronto or IrDA) available')
    else
    begin
      if (FCommunicationThread.State = CMachineStateSuspend) or
         (FCommunicationThread.State < 0) then
      begin
        FCommunicationThread.ProntoDataArray   := @ProntoDataArray;
        FCommunicationThread.ProntoMaxDataSize := sizeof(ProntoDataArray) div sizeof(Word);
        Caption := 'Waiting for remote control to be activated...';
        FCommunicationThread.State := CMachineStateIrlearn;
        repeat
          Application.ProcessMessages;
        until (FCommunicationThread.State = CMachineStateSuspend) or
              (FCommunicationThread.State < 0);
        if (FCommunicationThread.State = CMachineStateSuspend) then
        begin
          ProntoDataSize := FCommunicationThread.ProntoDataSize;
          InputString := ConvertProntoToASCII + #0;
          TheClipboard := Clipboard;
          TheClipboard.SetTextBuf(@InputString[1]);
          mmProntoData.Clear;
          if ProntoDataSize <> 0 then
            mmProntoData.Lines.Add(ConvertProntoToASCII);
          if btnMemoToClipboard.Enabled then
            btnMemoToClipboard.Enabled := False;
          Caption := 'Recording done';
        end
        else
        begin
          Caption := 'Recording failed';
          ProntoDataSize := 0;
        end;
      end
      else
        Caption := 'Device still busy';
    end;
  end
  else
  begin
    RecordingDataSize := 0;
    Caption := 'Waiting for remote control to be activated...';
    Refresh;
    // BUG?? timeout needs to be multiple of $1000 otherwise 'hangup'
    // We also have a 4 times smaller value then when we would use the VXD (timing is 4 times faster there)...
    if not(ReceiveIRDA(High(RecordingDataArray)-Low(RecordingDataArray),
                       @RecordingDataArray, Dword((tbTimeout.Position + 1) * $4000), BytesReturned)) then
    begin
      Caption := 'Timeout waiting for remote control';
    end
    else
    begin
      RecordingDataSize := BytesReturned + 1;
      ConvertResult := ConvertRecordingDataToProntoFormat(chkRepeat.Checked, ConvertMessage);
      case ConvertResult of
        0: InputString := ConvertProntoToASCII + #0;
        1: if chkAllowSuspiciousCode.Checked then
             InputString := ConvertProntoToASCII + #0
           else
           begin
             ProntoDataSize := 0;
             InputString := #0;
           end;
      else
        InputString := #0;
      end;
      Caption := ConvertMessage;
      TheClipboard := Clipboard;
      TheClipboard.SetTextBuf(@InputString[1]);
      mmProntoData.Clear;
      if ProntoDataSize <> 0 then
        mmProntoData.Lines.Add(ConvertProntoToASCII);
      if btnMemoToClipboard.Enabled then
        btnMemoToClipboard.Enabled := False;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Send data to IrDA device
            Source is the internal format
  Notes   :                                                                    
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnSendClick(Sender: TObject);
begin
  if RecordingDataSize <> 0 then
    SendInternalFormat
  else
    Caption := 'No data to send';
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Send data to IrDA device
            Source is the Pronto learned code format
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnSendProntoClick(Sender: TObject);
begin
  if ProntoDataSize <> 0 then
  begin
    if UsePronto then
    begin
      if FCommunicationThread = nil then
        ShowMessage('No destination (Pronto or IrDA) available')
      else
      begin
        if (FCommunicationThread.State = CMachineStateSuspend) or
           (FCommunicationThread.State < 0) then
        begin
          Caption := 'Sending data ...';
          FCommunicationThread.ProntoDataArray := @ProntoDataArray;
          FCommunicationThread.ProntoDataSize  := ProntoDataSize;
          FCommunicationThread.State := CMachineStateIrstart;
          repeat
            Application.ProcessMessages;
          until (FCommunicationThread.State = CMachineStateSuspend) or
            (FCommunicationThread.State < 0);
          if FCommunicationThread.State = CMachineStateSuspend then
            Caption := 'Sending done'
          else
            Caption := 'Sending failed';
        end
        else
          Caption := 'Device still busy';
      end
    end
    else
      SendProntoFormat;
  end
  else
    Caption := 'No data to send';
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Send Pronto data file to IrDA device
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.SendProntoFormat;
var
  UsCycles     : Extended;
  CTCClocks    : Byte;                           // Timer setting (assume a byte will fit)
  StartLoop    : Integer;                        // Start loop counter
  EndLoop      : Integer;                        // End loop counter
  Loop         : Integer;                        // Loop counter
  StateMachine : Integer;                        // State machine index
  State        : Integer;                        // On/Off loop
  RepeatCodes  : Integer;                        // Repeast to use
  Error        : Integer;
begin
  // Some minor tests
  Caption := 'Data contains invalid data';
  if ProntoDataSize <   6    then Exit;          // Must at least contain one ON/OFF sequence
  if ProntoDataArray[0] <> $0000 then Exit;      // Must be $0000
  if (ProntoDataArray[2] + ProntoDataArray[3]) <>
    ((ProntoDataSize div 2)-2) then Exit;        // Must be correct number of data

  Val(mskRepeats.EditText, RepeatCodes, Error);
  if Error <> 0 then
    Exit;

  Caption := 'Sending data ...';
  CheckIrDAInput;                                          // Necessary for example serial port to clear pending error otherwise no control possible
  Application.ProcessMessages;

  // Setup timer according to required resolution
  UsCycles := ProntoDataArray[1] * 0.241246;     // Frequency in us
  UsCycles := UsCycles / 0.838;                  // To timer units
  UsCycles := UsCycles / 2;                      // Because a complete carrier cycle is half ON and half OFF
  CTCClocks := round (UsCycles);
  CTCClocks := CTCClocks * 2;                    // Correction for mode 3 timer
  SetupTimedOutTimer(CtcClocks);

  StateMachine := 0;                             // Default state (exit)
  if ProntoDataArray[2] <> 0 then
    StateMachine := 1                            // State 1   = single
  else if ProntoDataArray[3] <> 0 then
         StateMachine := 2;                      // State 2.. = repeats

  while StateMachine <> 0 do
  begin
    // Determine start/end index for this run
    case StateMachine of
       1: begin
            StartLoop := 4;
            EndLoop   := StartLoop + (ProntoDataArray[2] * 2);
            if ProntoDataArray[3] = 0 then       // If no repeats indicated, we end
              StateMachine := 0
            else
              inc(StateMachine);
          end;
     else begin
            StartLoop := 4 + (ProntoDataArray[2] * 2);
            EndLoop   := StartLoop + (ProntoDataArray[3] * 2);
            inc(StateMachine);
            if StateMachine > (RepeatCodes+2) then
              StateMachine := 0;                 // After .. repeats always end
          end;
    end;

    // Do the number of indicated ON/OFF cycles
    Loop := StartLoop;
    while Loop < EndLoop do
    begin
      // This is the ON state were the carrier frequency is output
      for State := 0 to ProntoDataArray[Loop]-1 do
      begin
        ActivateIrDAOutput;
        WaitForTimedOut;
        DeactivateIrDAOutput;
        WaitForTimedOut;
      end;
      inc(Loop);
      // This is the OFF state were the carrier frequency is suppressed
      for State := 0 to ProntoDataArray[Loop]-1 do
      begin
        // Next is unnecessary but this way it is more understandable
        DeactivateIrDAOutput;
        WaitForTimedOut;
        DeactivateIrDAOutput;
        WaitForTimedOut;
      end;
      inc(Loop);
    end;
  end;
  DeactivateIrDAOutput;
  Caption := 'Sending done';
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Send internal format data to IrDA device
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.SendInternalFormat;
var
  Loop           : integer;                      // Loop counter
  OnOff          : longint;                      // Timer counter
  CTCClocks      : byte;                         // Timer setting (assume a byte will fit)
  Required       : extended;                     // Required timing
  Correction     : extended;                     // Correction for actual timing

begin
  // Some minor tests
  Caption := 'Data contains invalid data';
  if RecordingDataSize   <   6    then Exit;               // Must at least contain one ON/OFF sequence
  if not(RecordingDataArray[CType] in [2,4]) then Exit;    // Must be valid type

  Caption := 'Sending data ...';
  CheckIrDAInput;                                          // Necessary for example serial port to clear pending error otherwise no control possible
  Application.ProcessMessages;
  // Setup timer according to required resolution
  Required  := ((RecordingDataArray[CCycles] / RecordingDataArray[COperations]) * (RecordingDataArray[CCycleUnit]/1000)) / 0.838;
  CTCClocks := round (Required);
  if CTCClocks = 0 then CTCClocks := 1;
  Correction := Required / CTCClocks;
  CTCClocks := CTCClocks * 2;                              // Correction for mode 3 timer
  SetupTimedOutTimer(CtcClocks);

  Loop := CCycleUnit+1;
  while Loop < RecordingDataSize do
  begin
    if RecordingDataArray[CType]=2 then                    // If word source
    begin
      ActivateIrDAOutput;
      Required := RecordingDataArray[Loop] and $FFFF;
      Required := Correction * Required;
      for OnOff := 1 to round(Required) do
        WaitForTimedOut;
      DeactivateIrDAOutput;
      Required := (RecordingDataArray[Loop] shr 16) and $FFFF;
      Required := Correction * Required;
      for OnOff := 1 to round(Required) do
        WaitForTimedOut;
      inc(Loop);
    end
    else
    begin                                                  // If longint source
      ActivateIrDAOutput;
      Required := RecordingDataArray[Loop];
      Required := Correction * Required;
      for OnOff := 1 to round(Required) do
        WaitForTimedOut;
      inc(Loop);
      DeactivateIrDAOutput;
      if Loop >= RecordingDataSize then
      begin
        Caption := 'Invalid number of data';
        RecordingDataSize := 0;
        Exit;
      end;
      Required := RecordingDataArray[Loop];
      Required := Correction * Required;
      for OnOff := 1 to round(Required) do
        WaitForTimedOut;
      inc(Loop);
    end;
  end;
  DeactivateIrDAOutput;
  Caption := 'Sending done';
end;


{------------------------------------------------------------------------------
  Params  : <UseClipboard>  TRUE if clipboard is to be used
  Returns : <Result>        TRUE if success

  Descript: Generic load procedure. Load a file into memory. We detect the type
            we are dealing with automatically.
            The following types are possible:
            . internal format BINARY
              - identified by a leading 02 00  or  04 00           (BINARY)
            . internal format ASCII
              - identified by a leading 02 00  or  04 00           (ASCII)
            . Pronto format BINARY
              - identified by a leading 0000                       (BINARY)
            . Pronto format ASCII
              - identified by a leading 0000                       (ASCII)
              - identified by a leading 5000                       (ASCII RC5)
              - identified by a leading 5001                       (ASCII RC5X)
              - identified by a leading 6000                       (ASCII RC6)
              - identified by a leading 6001                       (ASCII RC6A)
            Data is loaded into the global data arrays. The indexes indicate
            if they are valid (non-zero).
  Notes   :
 ------------------------------------------------------------------------------}
function TfrmMain.LoadFile(UseClipboard: boolean): boolean;
type
  FileType = (ftUnknown, ftBinary, ftASCII, ftInternalBinary, ftInternalASCII, ftProntoBinary,
              ftProntoASCIILearned, ftProntoASCIIRC5, ftProntoASCIIRC5X, ftProntoASCIIRC6, ftProntoASCIIRC6A);
var
  TheFile      : file of byte;                             // File for input
  TheType      : FileType;                                 // Detected type
  AByte        : byte;                                     // Read byte
  InputString  : AnsiString;                               // Input is put here
  Part         : string;                                   // Partial string
  Value        : word;                                     // Conversion value
  Error        : integer;                                  // Error indicator
  Loop         : integer;                                  // Loop counter
  Bytes        : array[0..3] of byte;                      // Bytes -> longint
  ALongInt     : longint;                                  // Longint accu
  Reason       : string;                                   // Message
  TheClipboard : TClipboard;                               // Clipboard instance
  RCDelay      : word;                                     // Delay to use
  Calc         : extended;
  NewItemIndex : integer;
  ClickOrg     : TNotifyEvent;
  ConvertMessage: string;
begin
  Result  := False;
  TheType := ftUnknown;
  NewItemIndex := rgrpGenerateRC.ItemIndex;
  try
    Reason := 'Unknown';
    if UseClipboard then
    begin
      TheClipboard := Clipboard;                           // Get clipboard instance
      if TheClipboard.HasFormat(CF_TEXT) then
        InputString := TheClipboard.AsText
      else
        InputString := '';
    end
    else
    begin
      // Load the file
      if OpenDialog1.Execute then
      begin
        InputString := '';
        // Read in the file
        AssignFile(TheFile, OpenDialog1.FileName);
        Reset(TheFile);
        try
          while not(Eof(TheFile)) do
          begin
            Read(TheFile, AByte);
            InputString := InputString + chr(AByte);
          end;
        finally
          CloseFile(TheFile);
        end;
      end;
    end;
    Reason := 'Invalid input file';
    if length(InputString) < 2 then Exit;                  // Smallest size needed

    // Determine file type
    if InputString[1] < #5 then
      TheType := ftBinary
    else
      TheType := ftASCII;
    if (TheType = ftBinary) then
    begin
      // For binary we check the first two bytes only which MUST be 0 or 2 or 4
      Value := ord(InputString[1]) or (ord(InputString[2]) shl 8);
      if (Value in [2, 4]) then
        TheType := ftInternalBinary
      else if Value = 0 then
             TheType := ftProntoBinary
           else
             TheType := ftUnknown;
    end
    else
    begin
      // For ASCII we need to accumulate the first 4 valid characters
      // We only scan the first 12 character (which we always have!)
      Part := '';
      Loop := 1;
      repeat
        if ord(InputString[Loop]) in [ord('0')..ord('9'), ord('A')..ord('F'), ord('a')..ord('f')] then
          Part := Part + InputString[Loop];
        inc(Loop);
      until (length(Part) = 4) or (length(InputString) < Loop);
      if length(Part) < 4 then Part := 'FALSE';
      Part := '$' + Part;
      Val(Part, Value, Error);
      Value := Swap(Value);
      if (Value in [2, 4]) then
        TheType := ftInternalASCII
      else
      begin
        Value := Swap(Value);                              // Restore value
        case Value of
          $0000: TheType := ftProntoASCIILearned;
          $5000: TheType := ftProntoASCIIRC5;
          $5001: TheType := ftProntoASCIIRC5X;
          $6000: TheType := ftProntoASCIIRC6;
          $6001: TheType := ftProntoASCIIRC6A;
          else TheType := ftUnknown;
        end;
      end;
      if Error <> 0 then TheType := ftUnknown;
    end;
    if TheType = ftUnknown then Exit;

    // We now know the file type. According to this convert differently and in
    // the different global arrays.

    if TheType = ftInternalBinary then
    begin
      // BINARY INTERNAL FORMAT
      // Take each 4 bytes together
      RecordingDataSize := 0;
      while (length(InputString)<>0) do
      begin
        // Get 4 bytes
        for Loop := 0 to 3 do
        begin
          if length(InputString) = 0 then
          begin
            Reason := 'Error in length of file';
            RecordingDataSize := 0;
            Exit;
          end;
          Bytes[Loop] := ord(InputString[1]);
          Delete(InputString, 1, 1);
        end;
        ALongInt := (Bytes[3] shl 24) or (Bytes[2] shl 16) or (Bytes[1] shl 8) or Bytes[0];
        RecordingDataArray[RecordingDataSize] := ALongInt;
        Inc(RecordingDataSize);
        if RecordingDataSize > CRecordSize then
        begin
          Reason := 'Data file too long';
          RecordingDataSize := 0;
          Exit;
        end;
      end;
    end;

    if TheType = ftProntoBinary then
    begin
      // PRONTO BINARY FORMAT
      // Take each two bytes together
      ProntoDataSize := 0;
      while (length(InputString)<>0) do
      begin
        Value := ord(InputString[1]) shl 8;
        Delete(InputString, 1, 1);
        if length(InputString) = 0 then
        begin
          Reason := 'Error in length of file';
          ProntoDataSize := 0;
          Exit;
        end;
        AByte := ord(InputString[1]);
        Delete(InputString, 1, 1);
        Value := Value or AByte;
        ProntoDataArray[ProntoDataSize] := Value;
        Inc(ProntoDataSize);
        if ProntoDataSize > High(ProntoDataArray) then
        begin
          Reason := 'Data file too long';
          ProntoDataSize := 0;
          Exit;
        end;
      end;
    end;

    if TheType = ftInternalASCII then
    begin
      // ASCII INTERNAL FORMAT
      RecordingDataSize := 0;
      // First convert some special codes (CR/LF/TAB) into spaces
      while (Pos(#8 , InputString) <> 0) do InputString[Pos(#8 , InputString)] := ' ';
      while (Pos(#10, InputString) <> 0) do InputString[Pos(#10, InputString)] := ' ';
      while (Pos(#13, InputString) <> 0) do InputString[Pos(#13, InputString)] := ' ';

      while (length(InputString)<>0) do
      begin
        for Loop := 0 to 3 do
        begin
          Part := '';
          // Remove leading spaces
          while (length(InputString)<>0) and (InputString[1]=' ') do
            Delete(InputString, 1, 1);
          // Get all characters until a space or end of the string
          while (length(InputString)<>0) and (InputString[1]<>' ') do
          begin
            Part := Part + InputString[1];
            Delete(InputString, 1, 1);
          end;
          if Part<>'' then
          begin
            Part := '$' + Part;
            Val(Part, Bytes[Loop], Error);
            if Error <> 0 then
            begin
              Reason := 'Error in source file';
              RecordingDataSize := 0;
              Exit;
            end;
          end;
        end;
        ALongInt := (Bytes[3] shl 24) or (Bytes[2] shl 16) or (Bytes[1] shl 8) or Bytes[0];
        RecordingDataArray[RecordingDataSize] := ALongInt;
        Inc(RecordingDataSize);
        if RecordingDataSize > CRecordSize then
        begin
          Reason := 'Data file too long';
          RecordingDataSize := 0;
          Exit;
        end;
      end;
    end;

    // The RC5/RC5X/RC6/RC6A types are converted into the <ftProntoASCIILearned> type
    if TheType in [ftProntoASCIIRC5, ftProntoASCIIRC5X, ftProntoASCIIRC6, ftProntoASCIIRC6A] then
    begin
      // For all these types we have to 'extract' the Pronto data, i.e. 5000 0000 0000 0000 etc
      // to be converted to numbers
      // First convert some special codes (CR/LF/TAB) into spaces
      ProntoDataSize := 0;
      while (Pos(#8 , InputString) <> 0) do InputString[Pos(#8 , InputString)] := ' ';
      while (Pos(#10, InputString) <> 0) do InputString[Pos(#10, InputString)] := ' ';
      while (Pos(#13, InputString) <> 0) do InputString[Pos(#13, InputString)] := ' ';
      while (length(InputString)<>0) do
      begin
        Part := '';
        // Remove leading spaces
        while (length(InputString)<>0) and (InputString[1]=' ') do
          Delete(InputString, 1, 1);
        // Get all characters until a space or end of the string
        while (length(InputString)<>0) and (InputString[1]<>' ') do
        begin
          Part := Part + InputString[1];
          Delete(InputString, 1, 1);
        end;
        // Convert the value
        if Part<>'' then
        begin
          Part := '$' + Part;
          Val(Part, Value, Error);
          if Error <> 0 then
          begin
            Reason := 'Error in source file';
            ProntoDataSize := 0;
            Exit;
          end
          else
          begin
            ProntoDataArray[ProntoDataSize] := Value;
            Inc(ProntoDataSize);
            if ProntoDataSize > High(ProntoDataArray) then
            begin
              Reason := 'Data file too long';
              ProntoDataSize := 0;
              Exit;
            end;
          end;
        end;
      end;
    end;
    if TheType = ftProntoASCIIRC5 then
    begin
      // RC5 System Command
      //   System  [S]: 0..31
      //   Command [C]: 0..127
      // Pronto format (hex): 5000 0000 0000 0001 SSSS CCCC
      // We CAN NOT check on the exact amount of data because we will receive
      // padded '0000' from Pronto
      if ProntoDataSize < 6 then
      begin
        Reason := 'Incorrect RC5 data length';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[0] <> $5000) or
//         (ProntoData[1] <> $0000) or
         (ProntoDataArray[2] <> $0000) or
         (ProntoDataArray[3] <> $0001) then
      begin
        Reason := 'Incorrect RC5 data';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[4] > 31) or
         (ProntoDataArray[5] > 127) then
      begin
        Reason := 'Incorrect RC5 data (system/command)';
        ProntoDataSize := 0;
        Exit;
      end;
      // Everything correct sofar, now generate learned code
      if (ProntoDataArray[1] <> $0000)        and
          not chkFrequencyFixate.Checked then
      begin
        Calc := ProntoDataArray[1];
        Calc := Calc * 0.241246;
        Calc := 1E6 / Calc;
        Value := trunc(Calc) + 500;
        Value := (Value div 1000) * 1000;
        Loop := length(mskFrequency.EditText);
        mskFrequency.Text  := format('%*d', [Loop, Value]);
      end;
      NewItemIndex := 0;
      Loop := length(mskSystem.EditText);
      mskSystem.Text  := format('%*d', [Loop, ProntoDataArray[4]]);
      Loop := length(mskCommand.EditText);
      mskCommand.Text := format('%*d', [Loop, ProntoDataArray[5]]);
      val(mskDelay.EditText, RCDelay, Error);
      if Error <> 0 then RCDelay := CRCDelay;
      InputString := GenerateCodeRC5(chkToggle.Checked, ProntoDataArray[4], ProntoDataArray[5], RCDelay, chkRepeatPronto.Checked);
      TheType := ftProntoASCIILearned;                     // This is our new type now
    end;
    if TheType = ftProntoASCIIRC5X then
    begin
      // RC5X System Command Data
      //   System  [S]: 0..31
      //   Command [C]: 0..127
      //   Data    [D]: 0..63
      // Pronto format (hex): 5001 0000 0000 0002 SSSS CCCC DDDD 0000
      // We CAN NOT check on the exact amount of data because we will receive
      // padded '0000' from Pronto
      if ProntoDataSize < 8 then
      begin
        Reason := 'Incorrect RC5X data length';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[0] <> $5001) or
//         (ProntoData[1] <> $0000) or
         (ProntoDataArray[2] <> $0000) or
         (ProntoDataArray[3] <> $0002) or
         (ProntoDataArray[7] <> $0000) then
      begin
        Reason := 'Incorrect RC5X data';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[4] > 31) or
         (ProntoDataArray[5] > 127) or
         (ProntoDataArray[6] > 63) then
      begin
        Reason := 'Incorrect RC5X data (system/command/data)';
        ProntoDataSize := 0;
        Exit;
      end;
      // Everything correct sofar, now generate learned code
      if (ProntoDataArray[1] <> $0000)        and
          not chkFrequencyFixate.Checked then
      begin
        Calc := ProntoDataArray[1];
        Calc := Calc * 0.241246;
        Calc := 1E6 / Calc;
        Value := trunc(Calc) + 500;
        Value := (Value div 1000) * 1000;
        Loop := length(mskFrequency.EditText);
        mskFrequency.Text  := format('%*d', [Loop, Value]);
      end;
      NewItemIndex := 1;
      Loop := length(mskSystem.EditText);
      mskSystem.Text  := format('%*d', [Loop, ProntoDataArray[4]]);
      Loop := length(mskCommand.EditText);
      mskCommand.Text := format('%*d', [Loop, ProntoDataArray[5]]);
      Loop := length(mskData.EditText);
      mskData.Text := format('%*d', [Loop, ProntoDataArray[6]]);
      val(mskDelay.EditText, RCDelay, Error);
      if Error <> 0 then RCDelay := CRCDelay;
      InputString := GenerateCodeRC5X(chkToggle.Checked, ProntoDataArray[4], ProntoDataArray[5], ProntoDataArray[6], RCDelay, chkRepeatPronto.Checked);
      TheType := ftProntoASCIILearned;                     // This is our new type now
    end;
    if TheType = ftProntoASCIIRC6 then
    begin
      // RC6 System Command
      //   System  [S]: 0..255
      //   Command [C]: 0..255
      // Pronto format (hex): 6000 0000 0000 0001 SSSS CCCC
      // We CAN NOT check on the exact amount of data because we will receive
      // padded '0000' from Pronto
      if ProntoDataSize < 6 then
      begin
        Reason := 'Incorrect RC6 data length';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[0] <> $6000) or
//         (ProntoData[1] <> $0000) or
         (ProntoDataArray[2] <> $0000) or
         (ProntoDataArray[3] <> $0001) then
      begin
        Reason := 'Incorrect RC6 data';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[4] > 255) or
         (ProntoDataArray[5] > 255) then
      begin
        Reason := 'Incorrect RC6 data (system/command)';
        ProntoDataSize := 0;
        Exit;
      end;
      // Everything correct sofar, now generate learned code
      if (ProntoDataArray[1] <> $0000)        and
          not chkFrequencyFixate.Checked then
      begin
        Calc := ProntoDataArray[1];
        Calc := Calc * 0.241246;
        Calc := 1E6 / Calc;
        Value := trunc(Calc) + 500;
        Value := (Value div 1000) * 1000;
        Loop := length(mskFrequency.EditText);
        mskFrequency.Text  := format('%*d', [Loop, Value]);
      end;
      NewItemIndex := 2;
      Loop := length(mskSystem.EditText);
      mskSystem.Text  := format('%*d', [Loop, ProntoDataArray[4]]);
      Loop := length(mskCommand.EditText);
      mskCommand.Text := format('%*d', [Loop, ProntoDataArray[5]]);
      InputString := GenerateCodeRC6(False, ProntoDataArray[4], ProntoDataArray[5], CRCDelay, chkRepeatPronto.Checked);
      TheType := ftProntoASCIILearned;                     // This is our new type now
    end;
    if TheType = ftProntoASCIIRC6A then
    begin
      // RC6A CustumerCode System Command
      //   CustomerCode [U]: 0..127 or 32768..65535
      //   System       [S]: 0..255
      //   Command      [C]: 0..255
      // Pronto format (hex): 6001 0000 0000 0002 UUUU SSSS CCCC 0000
      // We CAN NOT check on the exact amount of data because we will receive
      // padded '0000' from Pronto
      if ProntoDataSize < 8 then
      begin
        Reason := 'Incorrect RC6A data length';
        ProntoDataSize := 0;
        Exit;
      end;
      if (ProntoDataArray[0] <> $6001) or
         (ProntoDataArray[1] <> $0000) or
         (ProntoDataArray[2] <> $0000) or
         (ProntoDataArray[3] <> $0002) or
         (ProntoDataArray[7] <> $0000) then
      begin
        Reason := 'Incorrect RC6A data';
        ProntoDataSize := 0;
        Exit;
      end;
      if ((ProntoDataArray[4] > 127) and (ProntoDataArray[4]<32768)) or
         (ProntoDataArray[5] > 255) or
         (ProntoDataArray[6] > 255) then
      begin
        Reason := 'Incorrect RC6A data (customer code/system/command)';
        ProntoDataSize := 0;
        Exit;
      end;
      // Everything correct sofar, now generate learned code
      if (ProntoDataArray[1] <> $0000)        and
          not chkFrequencyFixate.Checked then
      begin
        Calc := ProntoDataArray[1];
        Calc := Calc * 0.241246;
        Calc := 1E6 / Calc;
        Value := trunc(Calc) + 500;
        Value := (Value div 1000) * 1000;
        Loop := length(mskFrequency.EditText);
        mskFrequency.Text  := format('%*d', [Loop, Value]);
      end;
      NewItemIndex := 3;
      Loop := length(mskCustomerCode.EditText);
      mskCustomerCode.Text  := format('%*d', [Loop, ProntoDataArray[4]]);
      Loop := length(mskSystem.EditText);
      mskSystem.Text  := format('%*d', [Loop, ProntoDataArray[5]]);
      Loop := length(mskCommand.EditText);
      mskCommand.Text := format('%*d', [Loop, ProntoDataArray[6]]);
      InputString := GenerateCodeRC6A(False, ProntoDataArray[4], ProntoDataArray[5], ProntoDataArray[6], CRCDelay, chkRepeatPronto.Checked);
      TheType := ftProntoASCIILearned;                     // This is our new type now
    end;

    if TheType = ftProntoASCIILearned then
    begin
      // ASCII PRONTO FORMAT
      ProntoDataSize := 0;
      // First convert some special codes (CR/LF/TAB) into spaces
      while (Pos(#8 , InputString) <> 0) do InputString[Pos(#8 , InputString)] := ' ';
      while (Pos(#10, InputString) <> 0) do InputString[Pos(#10, InputString)] := ' ';
      while (Pos(#13, InputString) <> 0) do InputString[Pos(#13, InputString)] := ' ';

      while (length(InputString)<>0) do
      begin
        Part := '';
        // Remove leading spaces
        while (length(InputString)<>0) and (InputString[1]=' ') do
          Delete(InputString, 1, 1);
        // Get all characters until a space or end of the string
        while (length(InputString)<>0) and (InputString[1]<>' ') do
        begin
          Part := Part + InputString[1];
          Delete(InputString, 1, 1);
        end;
        // Convert the value
        if Part<>'' then
        begin
          Part := '$' + Part;
          Val(Part, Value, Error);
          if Error <> 0 then
          begin
            Reason := 'Error in source file';
            ProntoDataSize := 0;
            Exit;
          end
          else
          begin
            ProntoDataArray[ProntoDataSize] := Value;
            Inc(ProntoDataSize);
            if ProntoDataSize > High(ProntoDataArray) then
            begin
              Reason := 'Data file too long';
              ProntoDataSize := 0;
              Exit;
            end;
          end;
        end;
      end;
    end;

  finally
    Result := True;
    // Display error if such an error is detected
    if TheType = ftUnknown then Result := False;
    if (TheType in [ftInternalBinary, ftInternalASCII]) and
       (RecordingDataSize = 0)
      then Result := False;
    if (TheType in [ftProntoBinary, ftProntoASCIILearned, ftProntoASCIIRC5, ftProntoASCIIRC5X, ftProntoASCIIRC6, ftProntoASCIIRC6A]) and
       (ProntoDataSize = 0)
      then Result := False;
    if Result = False then Caption := Reason;
    if Result = True then
    begin
      if (TheType in [ftInternalBinary, ftInternalASCII]) then
      begin
//        Caption := 'Internal format data loaded';
        ConvertRecordingDataToProntoFormat(chkRepeat.Checked, ConvertMessage);
        Caption := ConvertMessage;
      end
      else
        Caption := 'Pronto format data loaded';
      mmProntoData.Clear;
      mmProntoData.Lines.Add(ConvertProntoToASCII);
      if btnMemoToClipboard.Enabled then
        btnMemoToClipboard.Enabled := False;
    end;
    // When we change the itemindex we MUST disable any 'click' event
    // because otherwise it is called 'recursively'
    if NewItemIndex <> frmMain.rgrpGenerateRC.ItemIndex then
    begin
      ClickOrg := rgrpGenerateRC.OnClick;
      rgrpGenerateRC.OnClick := nil;
      rgrpGenerateRC.ItemIndex := NewItemIndex;
      rgrpGenerateRC.Refresh;
      rgrpGenerateRC.OnClick := ClickOrg;
    end;
    // Now put data onto the clipboard
    InputString := ConvertProntoToASCII + #0;
    TheClipboard := Clipboard;
    TheCLipboard.SetTextBuf(@InputString[1]);
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Load file.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnLoadClick(Sender: TObject);
begin
  LoadFile(False);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Load from clipboard.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnLoadClipboardClick(Sender: TObject);
begin
  LoadFile(True);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Save internal format data to file (binary)
  Notes   :                                                                    
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnSaveInternalClick(Sender: TObject);
var
  TheFile : file of longint;                     { Output file }
  Loop    : integer;                             { Loop counter }
begin
  if (RecordingDataSize > 16) and
     (RecordingDataSize < CRecordSize) then
  begin
    if SaveDialog1.Execute then
    begin
      { Dump the detected timing data }
      AssignFile(TheFile, SaveDialog1.FileName);
      Rewrite(TheFile);
      try
        for Loop := 0 to RecordingDataSize-1 do
          Write(TheFile, RecordingDataArray[Loop]);
      finally
        CloseFile(TheFile);
      end;
      Caption := 'Data saved';
    end;
  end
  else Caption := 'Not saved';
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Save Pronto format data to file (ASCII)
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnSaveProntoClick(Sender: TObject);
var
  TheFile : file of byte;
  AByte   : byte;
  AString : AnsiString;
begin
  if (ProntoDataSize > 4) and
     (ProntoDataSize < High(ProntoDataArray)) then
  begin
    if SaveDialog1.Execute then
    begin
      AssignFile(TheFile, SaveDialog1.FileName);
      Rewrite(TheFile);
      try
        AString := ConvertProntoToASCII;
        while (length(AString)<>0) do
        begin
          AByte := ord(AString[1]);
          Delete(AString, 1, 1);
          Write(TheFile, AByte);
        end;
      finally
        CloseFile(TheFile);
      end;
      Caption := 'Data saved';
    end;
  end
  else Caption := 'Not saved';
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destruction of the form
  Notes   : -
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  FCommunicationThread.State := CMachineStateTerminate;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  LComPort: string;
  LAddress: Word;
  LError  : Integer;
begin
  UsePronto  := False;                                     // Use no PRONTO
  UseLPTPort := False;                                     // Use no LPT port
  UseSpecial := False;                                     // Use no special user defined port
  FCommunicationThread := nil;                             // Just to be sure
  // Depending on the command line parameter we use serial communication or the IrDA device
  // '1'..'9' will select COMx otherwise IrDA is used
  // By default, without a parameter, no hardware is used
  // As addition illegal options will alse default to no hardware used
  LComPort := '1';
  if ParamCount>0 then
  begin
    // A single number parameter indicates a communication port
    if (ParamCount = 1) then
    begin
      UsePronto := True;
      LComPort := ParamStr(1);
      // Any single parameter other than 1..9 indicates the LPT port
      if not (LComPort[1] in ['1'..'9']) then
      begin
        // LPT1 settings
        OutputPort := CLPT1 + CPrinterControl;
        OutputAnd  := CTransmitLine xor $FF;
        OutputOr   := CTransmitLine;
        OutputXor  := $00;  // or <OutputOr>

        InputPort  := CLPT1 + CPrinterStatus;
        InputAnd   := CReceiveLine;
        InputXor   := $00;  // or <InputAnd>
        UsePronto  := False;
        UseLPTPort := True;
      end;
    end;
    // Size parameters MIGHT indicate manual settings
    if (ParamCount = 6) then
    begin
      UseSpecial := True;                                 // Assume the parameters are all correct
      // Check HIGH/LOW
      if (Uppercase(ParamStr(3)) <> 'HIGH') and
         (Uppercase(ParamStr(3)) <> 'LOW') then
        UseSpecial := False;
      if (Uppercase(ParamStr(6)) <> 'HIGH') and
         (Uppercase(ParamStr(6)) <> 'LOW') then
        UseSpecial := False;
      // Check/set output port
      val(ParamStr(1), LAddress, LError);
      OutputPort := LAddress;
      if (LError <> 0) then
        UseSpecial := False;
      val(ParamStr(2), LAddress, LError);
      if (LError <> 0) then
        UseSpecial := False;
      OutputOr  := LAddress;
      OutputAnd := LAddress xor $FF;
      if (Uppercase(ParamStr(3)) = 'LOW') then
        OutputXor := OutputOr
      else
        OutputXor := $00;
      // Check/set input port
      val(ParamStr(4), LAddress, LError);
      InputPort := LAddress;
      if (LError <> 0) then
        UseSpecial := False;
      val(ParamStr(5), LAddress, LError);
      if (LError <> 0) then
        UseSpecial := False;
      InputAnd := LAddress;
      if (Uppercase(ParamStr(6)) = 'LOW') then
        InputXor := InputAnd
      else
        InputXor := $00;
    end;
  end;

  if UsePronto then
    FCommunicationThread := TCommunicationThread.Create(ord(LComPort[1])-$30);

  // Depending of what device we use we have to disable some stuff
  Caption := 'V3.02  No source/destination';
  grpRecord.Caption := 'No record source available';
  grpSend.Caption   := 'No sending destination available';
  if not UsePronto then
  begin
    if UseLPTPort then
      Caption := 'V3.02  Using IrDA device on parallel port'
    else
      if UseSpecial then
        Caption := format('V3.02  Using IrDA device on special ports (O:$%.3x, I:$%.3x)', [OutputPort, InputPort])
      else
        Caption := 'V3.02  Using no hardware';
    lblTimeout.Visible := True;
    tbTimeout.Visible  := True;
    tbTimeout.Enabled  := True;
    if UseLPTPort then
    begin
      grpRecord.Caption  := 'Record from IrDA (LPT)';
      grpSend.Caption    := 'Send to IrDA (LPT)';
    end
    else
    begin
      if UseSpecial then
      begin
        grpRecord.Caption  := 'Record from IrDA (user)';
        grpSend.Caption    := 'Send to IrDA (user)';
      end
      else
      begin
        // No hardware disables some button/items
        grpRecord.Visible  := False;
        grpSend.Visible    := False;
        lblTimeout.Visible := False;
        tbTimeout.Visible  := False;
        btnGenerateRC.Caption := 'Generate';
      end;
    end;
    if UseLPTPort or UseSpecial then
    begin
      DeactivateIrDAOutput;
      CheckIrDAInput;                                        // Necessary for example serial port to clear pending error otherwise no control possible
    end;  
  end;
  if FCommunicationThread <> nil then
  begin
    Caption := format('V3.02  Using Pronto on serial port %s', [LComPort[1]]);
    lblTimeout.Visible := False;
    tbTimeout.Visible  := False;
    tbTimeout.Enabled  := False;
    grpRecord.Caption  := 'Record from Pronto';
    grpSend.Caption    := 'Send to Pronto';
  end;
  RecordingDataSize := 0;                                  // No data available
  ProntoDataSize    := 0;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Change of RC format
  Notes   :                                                                    
 ------------------------------------------------------------------------------}
procedure TfrmMain.rgrpGenerateRCClick(Sender: TObject);
begin
  case rgrpGenerateRC.ItemIndex of
    0, 2:
       begin
         if lblCustomerCode.Visible then
           lblCustomerCode.Visible := False;
         if mskCustomerCode.Visible then
           mskCustomerCode.Visible := False;
         if lblData.Visible then
           lblData.Visible := False;
         if mskData.Visible then
           mskData.Visible := False;
       end;
    1:
       begin
         if lblCustomerCode.Visible then
           lblCustomerCode.Visible := False;
         if mskCustomerCode.Visible then
           mskCustomerCode.Visible := False;
         if not lblData.Visible then
           lblData.Visible := True;
         if not mskData.Visible then
           mskData.Visible := True;
       end;
    3:
       begin
         if not lblCustomerCode.Visible then
           lblCustomerCode.Visible := True;
         if not mskCustomerCode.Visible then
           mskCustomerCode.Visible := True;
         if lblData.Visible then
           lblData.Visible := False;
         if mskData.Visible then
           mskData.Visible := False;
       end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Generate RC code
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnGenerateRCClick(Sender: TObject);
var
  Value       : word;
  Error       : integer;
  Convert     : string;
  CustomerCode: word;
  System      : word;
  Command     : word;
  Data        : word;
  TheClipboard : TClipboard;                     // Clipboard instance
  RCDelay      : word;                           // Delay to use
  UseRepeat    : boolean;
begin
  CustomerCode := 0;
  Data         := 0;
  if mskCustomerCode.Visible then
  begin
    Convert := mskCustomerCode.EditText;
    Val(Convert, Value, Error);
    if Error<>0 then
    begin
      Caption := 'Incorrect Customer Code';
      Exit;
    end;
    if ((Value > 127) and (Value < 32768)) then
    begin
      Caption := 'Incorrect value Customer Code (0..127 or 32768..65535)';
      Exit;
    end;
    CustomerCode := Value;
  end;

  Convert := mskSystem.EditText;
  Val(Convert, Value, Error);
  if Error<>0 then
  begin
    Caption := 'Incorrect System';
    Exit;
  end;
  if ((Value > 63) and (rgrpGenerateRC.ItemIndex < 2)) then
  begin
    Caption := 'Incorrect value System (0..63)';
    Exit;
  end;
  if (Value > 255) then
  begin
    Caption := 'Incorrect value System (0..255)';
    Exit;
  end;
  System := Value;

  Convert := mskCommand.EditText;
  Val(Convert, Value, Error);
  if Error<>0 then
  begin
    Caption := 'Incorrect Command';
    Exit;
  end;
  if ((Value > 127) and (rgrpGenerateRC.ItemIndex < 2)) then
  begin
    Caption := 'Incorrect value Command (0..127)';
    Exit;
  end;
  if (Value > 255) then
  begin
    Caption := 'Incorrect value Command (0..255)';
    Exit;
  end;
  Command := Value;

  if mskData.Visible then
  begin
    Convert := mskData.EditText;
    Val(Convert, Value, Error);
    if Error<>0 then
    begin
      Caption := 'Incorrect Data';
      Exit;
    end;
    if (Value > 63) then
    begin
      Caption := 'Incorrect value Data (0..63)';
      Exit;
    end;
    Data := Value;
  end;

  TheClipboard := Clipboard;                               // Get clipboard instance
  // Generate code and place it on the clipboard
  val(mskDelay.EditText, RCDelay, Error);
  if Error <> 0 then RCDelay := CRCDelay;
  UseRepeat := chkRepeatRC.Checked;
  case rgrpGenerateRC.ItemIndex of
    0: TheClipboard.AsText := GenerateCodeRC5(chkToggle.Checked,  System, Command,       RCDelay, UseRepeat);
    1: TheClipboard.AsText := GenerateCodeRC5X(chkToggle.Checked, System, Command, Data, RCDelay, UseRepeat);
    2: TheClipboard.AsText := GenerateCodeRC6(chkToggle.Checked,  System, Command,       RCDelay, UseRepeat);
    3: TheClipboard.AsText := GenerateCodeRC6A(chkToggle.Checked, CustomerCode, System, Command, RCDelay, UseRepeat);
  end;
  LoadFile(True);                                          // Convert what's on the clipboard
//  SendProntoFormat;
  // If repeat >0 then manual repeating the code instead of just the repeat code indication
  if (chkRepeatRC.Checked) then
  begin
    if ((not UsePronto) and (not UseLPTPort) and (not UseSpecial)) or
        (FCommunicationThread <> nil) then
    begin
      ConvertToRepeatedCode;
      TheClipboard.AsText := ConvertProntoToASCII;
      LoadFile(True);
    end;
  end;
  if UsePronto or UseLPTPort or UseSpecial then
    btnSendProntoClick(Sender);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Copy memo to clipboard.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnMemoToClipboardClick(Sender: TObject);
var
  WholeMemo: AnsiString;
  TheClipboard : TClipboard;
begin
  WholeMemo := mmProntoData.Text + #0;
  TheClipboard := Clipboard;
  TheClipboard.SetTextBuf(@WholeMemo[1]);
  LoadFile(True);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Memo data changed.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.mmProntoDataChange(Sender: TObject);
begin
  if not btnMemoToClipboard.Enabled then
    btnMemoToClipboard.Enabled := True;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit.
  Notes   :
 ------------------------------------------------------------------------------}
initialization


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit.
  Notes   :
 ------------------------------------------------------------------------------}
finalization
  if UseLPTPort or UseSpecial then
  begin
    CheckIrDAInput;                                          // Necessary for example serial port to clear pending error otherwise no control possible
    DeactivateIrDAOutput;
  end;  
end.



