#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <stdlib.h>

#define MAXSAMPLE 30000

unsigned char data[MAXSAMPLE+1]; /* Recorded/read IR sample */
unsigned long speed; /* CPU speed factor */
int lpt; /* Printer port 1 or 2 */

#define INIFILE "IRREAD.INI"

#define BASE_8253 0x40
#define TIMER_0_8253 BASE_8253 + 0
#define MODE_8253 BASE_8253 + 3

#define PPORT1 (0x378)
#define STATUS1 (PPORT1+1)
#define PPORT2 (0x278)
#define STATUS2 (PPORT2+1)

static void my_delay(int val)
{
	static int x;
	x=val;
}

void StartTimer()
{
	outp( MODE_8253, 32+16+4 );
	my_delay(1);
	outp( TIMER_0_8253, 0 );
	my_delay(1);
	outp( TIMER_0_8253, 0 );
}

unsigned long StopTimer()
{
	unsigned int lo, hi, val;

	outp( MODE_8253, 0 );
	lo = inp( TIMER_0_8253 );
	my_delay(1);
	hi = inp( TIMER_0_8253 );
	val = (lo&255)|((hi&255)<<8);
	val = ~val;
	val += 1;
	
	return val;
}



/* Play a sample to IR port */
void Play( unsigned char *data, unsigned short length )
{
	unsigned int i, j;
	unsigned char mask, d;


	if (lpt==1) {	
		_disable();
		for (i=0; i<length; i++)
		{
			mask=128;
			d=data[i];
			for (j=0; j<8; j++)
			{
				if (d&mask) outp(PPORT1, 1);
				else outp(PPORT1,0);
				mask=mask>>1;
			}
		}
		_enable();
		outp(PPORT1, 0);
	} else {
		_disable();
		for (i=0; i<length; i++)
		{
			mask=128;
			d=data[i];
			for (j=0; j<8; j++)
			{
				if (d&mask) outp(PPORT2, 1);
				else outp(PPORT2,0);
				mask=mask>>1;
			}
		}
		_enable();
    	outp(PPORT2, 0);
    }

}


/* Determine the speed at which THIS CPU plays back codes */
unsigned long DeterminePlaySpeed( )
{
	unsigned int i, j, k;
	unsigned char mask, d;
	unsigned long speed;
	unsigned short length;
	char data[100];
	
	/* Clear out data memory, xmit nothing */
	for(i=0; i<100; i++) data[i] = 0;

	length = 100;
	speed = 0;
	
	_disable();
	for(k=0; k<16; k++) {
		StartTimer();
		for (i=0; i<length; i++)
		{
			mask=128;
			d=data[i];
			for (j=0; j<8; j++)
			{
				if (d&mask) outp(PPORT1, 1);
				else outp(PPORT1,0);
			
				mask=mask>>1;
			}
		}
		speed += StopTimer();
	}
	_enable();

	return speed;
}

float BitsPerUSec( unsigned long speed )
{
	float val;
	
	val = (float)800.0 /*bits*/ / ( (speed/(float)16.0) /*counts*/ * (float)0.8381 /*uS/count*/ ) ;
	
	return val;
}


void SynthesizeCarrier( unsigned long freq, unsigned long speed,
						unsigned char *data, unsigned short length )
{
	unsigned char mask, bit;
	float val;
	unsigned short fixed, signal, i, j;
	
	val = BitsPerUSec( speed ) * ((float)1000000.0/freq) /*uS*/;
	val = val * (float)0.5; /* Bits per toggle, 50% duty cycle */
	
	fixed = (unsigned short)(val * 256.0); /* Fixed point number math */
	
	signal=0;
	bit=1;
	for (i=0; i<length; i++){
		data[i]=0;
		mask = 128;
		for(j=0; j<8; j++) {
			if (bit) data[i] |= mask;
			mask = mask>>1;
			signal += 256;
			if (signal>fixed) {
				bit = (bit==0)?1:0;
				signal -= fixed;
			}
		}
	}
}


void Record( unsigned char *data, unsigned short *length )
{
	unsigned int i, j, len;
	unsigned char mask, d;
	

	if (lpt==1) {
		_disable();
		outp(PPORT1,2);
		while ((inp(STATUS1)&16)==0);
		while ((inp(STATUS1)&16)==16); /* Wait for 1st data */
	
		for (i=0; i<MAXSAMPLE; i++)
		{
			mask=128;
			d=0;
			for (j=0; j<8; j++)
			{
				if ((inp(STATUS1)&16)==0) d |= mask;
				mask=mask>>1;
			}
			data[i]=d;
		}
		outp(PPORT1, 0);
		_enable();
	} else {
		_disable();
    	outp(PPORT2,2);
		while ((inp(STATUS2)&16)==0);
		while ((inp(STATUS2)&16)==16); /* Wait for 1st data */

		for (i=0; i<MAXSAMPLE; i++)
		{
			mask=128;
			d=0;
			for (j=0; j<8; j++)
			{
				if ((inp(STATUS2)&16)==0) d |= mask;
				mask=mask>>1;
			}
			data[i]=d;
		}
		outp(PPORT2, 0);
		_enable();
	}

	/* Chop off ending zeros */
	for (len=MAXSAMPLE-1; len; len--)
		if (data[len]) break;

	*length = len;

}

unsigned long GuessFrequency( unsigned long speed, unsigned char *data, unsigned short length )
{
	double val, dt, xit;
	unsigned long trans;
	unsigned char mask, bit;
	unsigned short i, j;

	bit=0;
	trans=0;
	for (i=0; i<length; i++) {
		mask = 128;
		for (j=0; j<8; j++) {
			if ( ((bit==1) && ((mask&data[i])==0)) || ((bit==0) && ((mask&data[i])!=0)) ) {
				trans++;
				bit=(mask&data[i])?1:0;
			}
			mask = mask>>1;
		}
	}
	
	dt = (8.0 * length)/(BitsPerUSec( speed )*1000000.0);
	xit = 0.5 * trans;
	
	val = xit / dt;
	
	return (unsigned long)val;
}

void ParseSample( unsigned long freq, unsigned long speed,
				  unsigned char *data, unsigned short length,
				  FILE *fp )
{
	float val, bpus;
	unsigned short maxrun, counting;
	unsigned char mask, bit;
	unsigned short i, j;
	unsigned long cnt0, cnt1;
	
	/* Compute max # of consecutive 0s before considering EOC */
	bpus = BitsPerUSec(speed);
	val = ((float)3.0 / freq) * bpus * (float)1000000.0;
	maxrun = (unsigned short)val;
	
	cnt0=0;
	cnt1=0;
	counting = 1;
	bit=0;
	for(i=0; i<length; i++) {
		mask=128;
		for(j=0; j<8; j++) {
			if (counting==1) {
				cnt1++;
				bit = data[i]&mask;
				if (bit==0) {
					cnt0++;
					if (cnt0>maxrun) {
						cnt1 -= cnt0;
						val = ((float)cnt1)/bpus;
						if (val>1) fprintf(fp, "1:%ld uSec\n", (long)val);
						cnt1=0;
						counting=0;
					}
				} else
					cnt0=0;
			} else { /* counting==0 */
				cnt0++;
				bit = data[i]&mask;
				if (bit) {
					val = ((float)cnt0)/bpus;
					if (val>1) fprintf(fp, "0:%ld uSec\n", (long)val);
					cnt0=0;
					cnt1=1;
					counting=1;
				}
			}
		mask = mask>>1;
		}
	}

	val = ((float)((counting==0)?cnt0:cnt1))/bpus;
	fprintf(fp, "%d:%ld uSec\n", (int)counting, (long)val);
}


void ReadWaveformFile( FILE *fp, unsigned short wf[], unsigned short *idx,
                       unsigned long *carrier )
{
	int tmp;
	char line[256];
	int ctr;
	
	ctr=0;
	while( fgets(line, 255, fp) ) {
		if (line[0]!='#') {
			if (strncmp(line, "CARRIER", 7)==0) {
				sscanf(line, "%*s %ld", carrier);
			} else {
				sscanf(line, "%*c%*c%d", &tmp);
				wf[ctr] = tmp;
				ctr++;
			}
		}
	}
	wf[ctr]=0;
	*idx = ctr;
}


void GenerateWaveform( unsigned short wf[], unsigned long carrier,
                       unsigned long speed, unsigned char data[], unsigned short *size )
{
	float bpus;
	unsigned short idx, loc;
	unsigned long span;
	unsigned char flag;

	bpus = BitsPerUSec( speed );
	SynthesizeCarrier( carrier, speed, data, *size );
	idx=0;
	loc=0;
	flag=255;
	while (span=wf[idx]) {
		span = (unsigned long)(span /*uSec*/ * bpus /*bits/uSec*/ / 8 /*bits/byte*/);
		for(; span; loc++, span--)
			data[loc] &= flag;
		flag ^= 255;
		idx++;
	}
	*size = loc;
}



/* Search for 3 bytes of zeros... */
unsigned short PreambleLength( unsigned char *data, unsigned short length )
{
	unsigned short i;
	int cnt;
	
	cnt=0;
	for (i=0; i<length; i++) {
		if (*data==0) cnt++;
		else cnt=0;
		if (cnt==3) return i-3;
		data++;
	}
	return length;
}



void ReadINIFile( char *fname, unsigned long *speed, int *lpt )
{
	FILE *fp;
	char line[256];
	
	/* Sane defaults */
	*speed = 25584;
	*lpt = 1;
	
	/* Read File */
	fp=fopen(fname, "r");
	if (fp==NULL) return;
	while ( fgets(line, 255, fp) ) {
		if ( (strncmp(line, "SPEED =", 7)==0)||(strncmp(line, "speed =", 7)==0) ) {
			sscanf(line, "%*s%*c%*c%ld", speed);
			if (*speed==0) *speed = 25584;
		} else if ( (strncmp(line, "LPT =", 5)==0)||(strncmp(line, "lpt =", 5)==0) ) {
			sscanf(line, "%*s%*c%*c%d", lpt);
			if ((*lpt>2)||(*lpt<1)) *lpt = 1;
		}
	}
	fclose(fp);
}


void WriteINIFile()
{
	FILE *fp;
	fp = fopen( INIFILE, "w" );
	
	if (fp==NULL) return;
	fprintf(fp, "SPEED = %ld\n", speed);
	fprintf(fp, "LPT = %d\n", lpt);
	fclose(fp);
}


void DoCPUSpeed()
{
	unsigned long s1;
	char c;
		
	printf("Computing CPU Speed...\n");
	s1 = DeterminePlaySpeed();
	printf("Computed CPU Speed: %ld\n", s1);
	printf("Accept this speed (Y/N)?");
	do { scanf("%c", &c); } while ((tolower(c)!='y')&&(tolower(c)!='n'));
	if (tolower(c)=='n') {
		printf("Old CPU Speed: %ld\n", speed);
		printf("New CPU Speed: ");
		scanf("%ld", &s1);
	}
	if (s1==0) s1=25000;
	speed = s1;
	printf("Write to INI file (Y/N)?");
	do { scanf("%c", &c); } while ((tolower(c)!='y')&&(tolower(c)!='n'));
	if (tolower(c)=='y') WriteINIFile();
}


void DoLPTPort()
{
	char c;
		
	lpt = (lpt==1)?2:1;
	printf("New Printer Port: %d\n", lpt);
	printf("Write to INI file (Y/N)?");
	do { scanf("%c", &c); } while ((tolower(c)!='y')&&(tolower(c)!='n'));
	if (tolower(c)=='y') WriteINIFile();
}
	

void DoBinaryRecord()
{
	unsigned short length;
	char fname[256];
	FILE *fp;
	
	printf("\n\nPlace remote 3-6\" in front of the sensor and press the button to record.\n");
	printf("Waiting for code...\n");
	Record( data, &length );
	printf("Code acquired.  Length = %ld bits\n", (long) (8L*(long)length) );
	printf("BINARY file to save as: ");
	scanf("%s", fname);
	fp=fopen(fname, "wb");
	if (fp) {
		fwrite(data, length, 1, fp);
		fclose(fp);
	} else
		printf("ERROR!  Unable to open file for writing!\n\n");
	
	printf("\n\n");
}

void DoBinaryPlay()
{
	unsigned short length;
	char fname[256];
	FILE *fp;
	
	printf("BINARY file to play: ");
	scanf("%s", fname);
	fp=fopen(fname, "rb");
	if (fp==NULL) {
		printf("ERROR!  Unable to open file for reading!\n\n");
		return;
	}
	length = fread(data, 1, MAXSAMPLE-1, fp); 
	fclose(fp);
	printf("Code loaded.  Length = %ld bits\n", (long)(8L*(long)length) );
	printf("Playing code...\n");
	Play(data, length);
	printf("Completed.\n\n");
}



void DoTextRecord()
{
	unsigned short length, plength;
	unsigned long freq;
	char fname[256];
	FILE *fp;
	struct tm *newtime;
	time_t aclock;

	printf("\n\nPlace remote 3-6\" in front of the sensor and press the button to record.\n");
	printf("Waiting for code...\n");
	Record( data, &length );
	printf("Code acquired.  Length = %ld bits\n", (long) (8L*(long)length) );
	printf("TEXT file to save as: ");
	scanf("%s", fname);
	fp=fopen(fname, "w");
	if (fp) {
		printf("Decoding signal...\n");
		time( &aclock );                 /* Get time in seconds */
		newtime = localtime( &aclock );  /* Convert time to struct tm form  */
		fprintf(fp, "# IRREAD by Earle F. Philhower, III\n");
		fprintf(fp, "# http://www.geocities.com/CapeCanaveral/6552\n");
		fprintf(fp, "# Sample recorded on: %s\n", asctime( newtime ) );
		plength = PreambleLength(data, length);
		printf("Preamble: %d bytes...\n", (int)plength);
		printf("Frequency: %ld Hertz\n", (unsigned long)freq=GuessFrequency(speed, data, plength) );
		fprintf( fp, "CARRIER %ld\n", freq );
		ParseSample( freq, speed, data, length, fp );
		fclose(fp);
		printf("Wrote file.\n");
	} else
		printf("ERROR!  Unable to open file for writing!\n\n");
	
	printf("\n\n");
}

void DoTextPlay()
{
	unsigned short length;
	char fname[256];
	unsigned short wf[256];
	unsigned short wfsize;
	FILE *fp;
	unsigned long freq;
	
	printf("TEXT file to play: ");
	scanf("%s", fname);
	fp=fopen(fname, "r");
	if (fp==NULL) {
		printf("ERROR!  Unable to open file for reading!\n\n");
		return;
	}
	printf("Reading waveform file...\n");
    ReadWaveformFile(fp, wf, &wfsize, &freq);
	printf("Frequency: %ld Hertz\n", freq);
	fclose(fp);
	length=MAXSAMPLE-1;
	printf("Synthesizing waveform...\n");
	GenerateWaveform(wf, freq, speed, data, &length);
	printf("Final length: %ld bits\n", (long)(8L*(long)length));
	printf("Playing waveform...\n");
	Play(data, length);
	printf("Completed.\n\n");
}


void BatchBinaryPlay( char *fname )
{
	unsigned short length;
	FILE *fp;
	
	fp=fopen(fname, "rb");
	if (fp==NULL) {
		fprintf(stderr, "ERROR!  Unable to open \"%s\" for reading!\n\n", fname);
		return;
	}
	length = fread(data, 1, MAXSAMPLE-1, fp); 
	fclose(fp);
	Play(data, length);
}

void BatchTextPlay( char *fname )
{
	unsigned short length;
	unsigned short wf[256];
	unsigned short wfsize;
	FILE *fp;
	unsigned long freq;
	
	fp=fopen(fname, "r");
	if (fp==NULL) {
		fprintf(stderr, "ERROR!  Unable to open \"%s\" for reading!\n\n", fname);
		return;
	}
    ReadWaveformFile(fp, wf, &wfsize, &freq);
	fclose(fp);
	length=MAXSAMPLE-1;
	GenerateWaveform(wf, freq, speed, data, &length);
	Play(data, length);
}




int main(int argc, char *argv[])
{
	int i;
	int bad;

	ReadINIFile( INIFILE, &speed, &lpt );

	if (argc-1) {
		bad=1;
		lpt=1;
		for(i=0; i<argc; i++) {
			if (strcmp(argv[i+1], "-speed")==0) {
				speed = atol( argv[i+2] );
				if (speed==0) speed=25000;
				i++;
				bad=0;
			} else if (strcmp(argv[i+1], "-lpt1")==0) {
				lpt=1;
				bad=0;
			} else if (strcmp(argv[i+1], "-lpt2")==0) {
				lpt=2;
				bad=0;
			} else if (strcmp(argv[i+1], "-binary")==0) {
				BatchBinaryPlay( argv[i+2] );
				i++;
				bad=0;
			} else if (strcmp(argv[i+1], "-text")==0) {
				BatchTextPlay( argv[i+2] );
				i++;
				bad=0;
			} else {
				if (bad) {
					printf("ERROR!  Usage: %s [options]\n", argv[0]);
					printf("No options brings up a menu system\n");
					printf("-speed xxxx sets speed to xxxx\n");
					printf("-lpt1 -lpt2 sets port to xxxx\n");
					printf("-binary xxxx plays BINARY IR file\n");
					printf("-text xxxx plays TEXT IR file\n");
					return -1;
				} else {
					return 0;
				}
			}
		}
		return 0;
	}

	while(1) {
		printf("Menu\n\n");
		printf("1. Determine CPU Speed (under DOS ONLY!)\n");
		printf("   (Current speed = %ld)\n", speed);
		printf("2. Toggle Parallel Port:  Current=%s\n\n",(lpt==2)?"LPT2":"LPT1" );
		printf("3. Record Binary IR Sample\n");
		printf("4. Playback Binary IR Sample\n\n");
		printf("5. Record Textual IR Sample\n");
		printf("6. Playback Textual IR Sample\n\n");
		printf("7. Exit\n\n");
		printf("Choice: ");
		scanf("%d", &i);
		
		switch(i) {
			case 1: DoCPUSpeed(); break;
			case 2: DoLPTPort(); break;
			case 3: DoBinaryRecord(); break;
			case 4: DoBinaryPlay(); break;
			case 5: DoTextRecord(); break;
			case 6: DoTextPlay(); break;
			case 7: return 0; break;
		}
	}
}
